package support.andro.app.rfid.adapter;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import com.asreader.app.rfid.R;
import com.asreader.p252b.diagnostics.AsReaderP252BLog;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.LinearLayout;
import android.widget.TextView;

public class TagListAdapter extends BaseAdapter {

	private static final String TAG = TagListAdapter.class.getSimpleName();
	private static final int UPDATE_TIME = 300;

	private LayoutInflater mInflater;
	private ArrayList<TagListItem> mList;
	private HashMap<String, TagListItem> mMap;
	private boolean mIsDisplayPc;
	private boolean mIsReportRSSI;
	private boolean mIsShowCount;
	private boolean mIsReportTID;
	private Handler mHandler;
	private UpdateThread mUpdateThread;

	// ------------------------------------------------------------------------
	// Constructor
	// ------------------------------------------------------------------------
	public TagListAdapter(Context context) {
		super();

		mInflater = (LayoutInflater) context
				.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		mList = new ArrayList<>();
		mMap = new HashMap<>();
		mIsDisplayPc = true;
		mIsShowCount = true;
		mIsReportRSSI = false;
		mIsReportTID = false;
		mUpdateThread = null;
		mHandler = new Handler();
	}

	public void clear() {
		mList.clear();
		mMap.clear();
		Handler mainHandler = new Handler(Looper.getMainLooper());
		mainHandler.post (new Runnable() {
			@Override
			public void run() {
				notifyDataSetChanged();
			}
		});
	}

	@SuppressWarnings("unused")
	public boolean getDisplayPc() {
		return mIsDisplayPc;
	}

	public void setDisplayPc(boolean enabled) {
		mIsDisplayPc = enabled;
		notifyDataSetChanged();
	}

	@SuppressWarnings("unused")
	public boolean getShowCount() {
		return mIsShowCount;
	}

	@SuppressWarnings("unused")
	public void setShowCount(boolean enabled) {
		mIsShowCount = enabled;
		notifyDataSetChanged();
	}

	@SuppressWarnings("unused")
	public boolean getReportRSSI() {
		return mIsReportRSSI;
	}
	
	public void setReportRSSI(boolean enabled) {
		if (mIsReportRSSI == enabled)
			return;
		mIsReportRSSI = enabled;
		clear();
	}

	@SuppressWarnings("unused")
	public boolean getReportTID() {
		return mIsReportTID;
	}

	@SuppressWarnings("unused")
	public void setReportTID(boolean enabled) {
		if (mIsReportTID == enabled)
			return;
		mIsReportTID = enabled;
		clear();
	}

	public void start() {
		mUpdateThread = new UpdateThread();
		mUpdateThread.start();
	}

	public void shutDown() {
		if (mUpdateThread != null) {
			mUpdateThread.cancel();
			try {
				mUpdateThread.join();
			} catch (InterruptedException e) {
				AsReaderP252BLog.e(TAG, "ERROR. shutDown mUpdateThread.join() - [%s]", e.getMessage());
			}
			mUpdateThread = null;
		}
	}

	public void addTag(String tag, float rssi, float phase, float frequency, long time) {
		TagListItem item;

		if (null == (item = mMap.get(tag))) {
			item = new TagListItem(tag, rssi, phase, frequency, time);
			mList.add(item);
			mMap.put(tag, item);
		} else {
			item.updateItem(rssi, phase, frequency, time);
		}
	}

	public void addTag(String tag, String tid, float rssi, float phase, float frequency, long time) {
		TagListItem item;

		if (null == (item = mMap.get(tag))) {
			item = new TagListItem(tag, tid, rssi, phase, frequency, time);
			mList.add(item);
			mMap.put(tag, item);
		} else {
			if (item.getTid().equals(tid))
				item.updateItem(rssi, phase, frequency, time);
			else {
				item = new TagListItem(tag, tid, rssi, phase, frequency, time);
				mList.add(item);
				mMap.put(tag, item);
			}
		}
	}

	@Override
	public int getCount() {
		return mList.size();
	}

	@Override
	public String getItem(int position) {
		return mList.get(position).getTag(true);
	}
	
	public String getItem(int position, boolean displayPc) {
		return mList.get(position).getTag(displayPc);
	}
	
	public String getItemForCsv(int position) {
		return getItemForCsv(position, true);
	}
	public String getItemForCsv(int position, boolean displayPc)
	{
		TagListItem item = mList.get(position);
		return String.format(Locale.US, "%s,%s,%d", item.getTag(displayPc), item.getTime(), item.getCount());
	}

	@Override
	public long getItemId(int position) {
		return position;
	}

	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		TagListViewHolder holder;

		if (null == convertView) {
			convertView = mInflater.inflate(R.layout.item_tag_list, parent,
					false);
			holder = new TagListViewHolder(convertView);
		} else {
			holder = (TagListViewHolder) convertView.getTag();
		}
		holder.setItem(mList.get(position), mIsDisplayPc);
		return convertView;
	}

	// ------------------------------------------------------------------------
	// Internal TagListItem class
	// ------------------------------------------------------------------------

	private class TagListItem {

		private String mTag;
		private String mTid;
		private float mRSSI;
		private float mPhase;
		private float mFrequency;
		private int count;
		private long mTime;
		
		private SimpleDateFormat mDateFormat;

		@SuppressWarnings("unused")
		public TagListItem(String tag) {
			mTag = tag;
			mTid = "";
			mRSSI = 0;
			mPhase = 0;
			mFrequency = 0;
			count = 1;
			mTime = System.currentTimeMillis();
			
			mDateFormat = new SimpleDateFormat("yyy/MM/dd HH:mm:ss", Locale.US);
		}

		private TagListItem(String tag, float rssi, float phase, float frequency, long time) {
			mTag = tag;
			mTid = "";
			mRSSI = rssi;
			mPhase = phase;
			mFrequency = frequency;
			count = 1;
			mTime = time;
			
			mDateFormat = new SimpleDateFormat("yyy/MM/dd HH:mm:ss", Locale.US);
		}

		private TagListItem(String tag, String tid, float rssi, float phase, float frequency, long time) {
			mTag = tag;
			mTid = tid;
			mRSSI = rssi;
			mPhase = phase;
			mFrequency = frequency;
			count = 1;
			mTime = time;
			
			mDateFormat = new SimpleDateFormat("yyy/MM/dd HH:mm:ss", Locale.US);
		}

		public String getTag(boolean displayPc) {
			return displayPc ? mTag : mTag.substring(4);
		}

		private String getTid() {
			return mTid;
		}

		private float getRSSI() {
			return mRSSI;
		}
		
		public float getPhase() {
			return mPhase;
		}
		
		public float getFrequency() {
			return mFrequency;
		}

		public int getCount() {
			return count;
		}
		
		public String getTime() {
			return mDateFormat.format(new Date(mTime));
		}

		private void updateItem(float rssi, float phase, float frequency, long time) {
			mRSSI = rssi;
			mPhase = phase;
			mFrequency = frequency;
			count++;
			mTime = time;
		}
	}

	// ------------------------------------------------------------------------
	// Internal TagListViewHodler class
	// ------------------------------------------------------------------------
	private class TagListViewHolder {
		private TextView txtTag;
		private TextView txtTid;
		private TextView txtRssi;
		private TextView txtPhase;
		private TextView txtFrequency;
		private TextView txtCount;
		private LinearLayout layoutSubItems;
		
		private float nSize = 13;
		private float nDecrease = (float) 0.3;

		private TagListViewHolder(View parent) {
			txtTag = (TextView) parent.findViewById(R.id.tag_value);
			txtTid = (TextView) parent.findViewById(R.id.tid_value);
			txtTid.setVisibility(mIsReportTID ? View.VISIBLE : View.GONE);
			txtRssi = (TextView)parent.findViewById(R.id.tag_rssi);
			txtPhase = (TextView)parent.findViewById(R.id.tag_phase);
			txtFrequency = (TextView)parent.findViewById(R.id.tag_frequency);
			txtCount = (TextView) parent.findViewById(R.id.tag_count);
			txtCount.setVisibility(mIsShowCount ? View.VISIBLE : View.GONE);
			layoutSubItems = (LinearLayout)parent.findViewById(R.id.sub_items);
			layoutSubItems.setVisibility(mIsReportRSSI ? View.VISIBLE : View.GONE);
			parent.setTag(this);
		}

		public void setItem(TagListItem item, boolean displayPc) {
			txtTag.setText(item.getTag(displayPc));
			txtTid.setText(item.getTid());
			txtRssi.setText(String.format(Locale.US, "%.1f dB", item.getRSSI()));
			txtPhase.setText(String.format(Locale.US, "%.1f˚", item.getPhase()));
			txtFrequency.setText(String.format(Locale.US, "%.2fMHz", item.getFrequency()));
			int cnt = item.getCount();
			if(cnt > 9999999) {
				txtCount.setTextSize(TypedValue.COMPLEX_UNIT_SP, (nSize - (nDecrease * 12)));
			} else if(cnt > 999999) {
				txtCount.setTextSize(TypedValue.COMPLEX_UNIT_SP, (nSize - (nDecrease * 8)));
			} else if(cnt > 99999) {
				txtCount.setTextSize(TypedValue.COMPLEX_UNIT_SP, (nSize - (nDecrease * 4)));
			} else if(cnt > 9999) {
				txtCount.setTextSize(TypedValue.COMPLEX_UNIT_SP, (nSize - nDecrease));
			}
			txtCount.setText(String.format(Locale.US, "%d", item.getCount()));
			layoutSubItems.setVisibility(mIsReportRSSI ? View.VISIBLE : View.GONE);
		}
	}

	// ------------------------------------------------------------------------
	// Internal UpdateThread class
	// ------------------------------------------------------------------------
	private class UpdateThread extends Thread {

		private boolean mIsAlived;

		private UpdateThread() {
			mIsAlived = false;
		}

		@Override
		public void run() {
			mIsAlived = true;
			
			while (mIsAlived) {
				synchronized (TagListAdapter.this) {
					try {
						TagListAdapter.this.wait(UPDATE_TIME);
					} catch (InterruptedException e) {
						AsReaderP252BLog.e(TAG, "ERROR. UpdateThread, TagListAdapter.this.wait(%d) - [%s]",
								UPDATE_TIME, e.getMessage());
					}
				}
				
				if (mIsAlived) {
					mHandler.post(procUpdate);
				}
			}
		}
		
		private Runnable procUpdate = new Runnable() {

			@Override
			public void run() {
				TagListAdapter.this.notifyDataSetChanged();
			}
			
		};

		private void cancel() {
			synchronized (TagListAdapter.this) {
				mIsAlived = false;
				TagListAdapter.this.notify();
			}
		}
	}
}
