package support.andro.app.rfid.view.base;

import java.util.Locale;


import support.andro.app.rfid.adapter.PowerRangeAdapter;
import support.andro.app.rfid.filter.InputFilterMinMax;
import support.andro.app.rfid.util.ResUtil;

import com.asreader.app.rfid.MainActivity;
import com.asreader.p252b.diagnostics.AsReaderP252BLog;
import com.asreader.p252b.rfid.exception.AsReaderP252BException;
import com.asreader.p252b.rfid.params.AsReaderP252BPowerRange;
import com.asreader.app.rfid.R;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.text.InputFilter;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

public abstract class RfidPowerGainActivity extends RfidClearActivity {

	private static final String TAG = RfidPowerGainActivity.class.getSimpleName();

	private static final int DEFAULT_POWER_GAIN = 300;

	private TextView txtPower;
	private TextView txtOperationTime;

	private int mPowerGain;
	private int mOperationTime;
	private AsReaderP252BPowerRange mAsReaderP252BPowerRange;

	protected boolean mIsEnabled;

	@Override
	protected void createReader() {
		super.createReader();
		mPowerGain = DEFAULT_POWER_GAIN;
		mAsReaderP252BPowerRange = getReader().getPowerGainRange();
		mIsEnabled = false;

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. createReader()");
	}

	// Initialize Reader
	@Override
	protected void initReader() {
		int power;
		int time;

		// Get Power Gain
		try {
			power = getReader().getPowerGain();
		} catch (AsReaderP252BException e) {
			power = mAsReaderP252BPowerRange.getMax();
		}
		setPowerGain(power);
		// Get Operation Time
		try {
			time = getReader().getOperationTime();
		} catch (AsReaderP252BException e) {
			AsReaderP252BLog.e(TAG, e, "ERROR. initReader() - Failed to get operation time");
			time = 0;
		}
		setOperationTime(time);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. initReader()");
	}

	@Override
	public void onClick(View v) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onClick(%s)", ResUtil.getId(v.getId()));

		switch (v.getId()) {
		case R.id.power_gain:
			showPowerGainDialog();
			break;
		case R.id.operation_time:
			showOperationTimeDialog();
			break;
		}
		super.onClick(v);
	}

	protected void initWidgets() {
		super.initWidgets();

		// Power Gain Spinner
		txtPower = (TextView) findViewById(R.id.power_gain);
		if(txtPower != null)
			txtPower.setOnClickListener(this);
		// Operation Time EditText
		txtOperationTime = (TextView) findViewById(R.id.operation_time);
		if(txtOperationTime != null)
			txtOperationTime.setOnClickListener(this);
		setOperationTime(mOperationTime);
		
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. initWidgets()");
	}

	protected void enableActionWidgets(boolean enabled) {
		super.enableActionWidgets(enabled);

		mIsEnabled = isEnabledWidget(enabled);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. enableActionWidgets(%s)", enabled);
	}

	protected void setPowerGain(int power) {
		
		if(txtPower == null)
			return;
		
		mPowerGain = power;
		txtPower.setText(String.format(Locale.US, "%.1f dBm", (double) power / 10.0));

		try {
			getReader().setPowerGain(mPowerGain);
			MainActivity.mUserConfigs.setPowerGain(mPowerGain);
			MainActivity.mUserConfigs.Save();
		} catch (AsReaderP252BException e) {
			AsReaderP252BLog.e(TAG, "ERROR. setPowerGain(%d) - Failed to set power gain", mPowerGain);
			return;
		}
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setPowerGain(%d)", power);
	}

	@SuppressWarnings("unused")
	protected int getPowerGain() {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. getPowerGain() - [%d]", mPowerGain);
		return mPowerGain;
	}

	protected void setOperationTime(int time) {
		if(txtOperationTime == null)
			return;
		
		mOperationTime = time;
		txtOperationTime.setText(String.format(Locale.US, "%d s", mOperationTime));
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setOperationTime(%d)", time);
	}

	protected int getOperationTime() {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. getOperationTime() - [%d]", mOperationTime);
		return mOperationTime;
	}

	private void showPowerGainDialog() {

		if (!mIsEnabled)
			return;

		LinearLayout root = (LinearLayout) LinearLayout.inflate(this, R.layout.dialog_list_view, null);
		final ListView power = (ListView) root.findViewById(R.id.list);
		final PowerRangeAdapter adapter = new PowerRangeAdapter(this, mAsReaderP252BPowerRange,
				android.R.layout.simple_list_item_single_choice);
		power.setAdapter(adapter);
		power.setChoiceMode(ListView.CHOICE_MODE_SINGLE);

		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setTitle(R.string.power_gain);
		builder.setView(root);
		builder.setPositiveButton(R.string.action_ok, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				int position = power.getCheckedItemPosition();
				int power = adapter.getItemValue(position);
				setPowerGain(power);
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showPowerGinaDialog().$PositiveButton.onClick()");
			}
		});
		builder.setNegativeButton(R.string.action_cancel, new DialogInterface.OnClickListener() {
			
			@Override
			public void onClick(DialogInterface dialog, int which) {
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showPowerGinaDialog().$NegativeButton.onClick()");
			}
		});
		builder.setCancelable(true);
		builder.setOnCancelListener(new DialogInterface.OnCancelListener() {
			
			@Override
			public void onCancel(DialogInterface dialog) {
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showPowerGinaDialog().onCancel()");
			}
		});
		AlertDialog dialog = builder.create();
		dialog.setOnShowListener(new DialogInterface.OnShowListener() {

			@Override
			public void onShow(DialogInterface dialog) {
				int position = adapter.getItemPosition(mPowerGain);
				power.setItemChecked(position, true);
				power.setSelectionFromTop(position, 0);
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showPowerGinaDialog().onShow()");
			}
		});
		dialog.show();

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showPowerGinaDialog()");
	}

	private void showOperationTimeDialog() {

		if (!mIsEnabled)
			return;

		LinearLayout root = (LinearLayout) LinearLayout.inflate(this, R.layout.dialog_input_unit, null);
		final EditText operationTime = (EditText) root.findViewById(R.id.value);

		final TextView unit = (TextView) root.findViewById(R.id.unit);
		unit.setText(R.string.unit_sec);
		final EditText value = (EditText) root.findViewById(R.id.value);
		InputFilter[] filters = new InputFilter[] { new InputFilterMinMax(0, 1800) };
		value.setFilters(filters);

		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setTitle(R.string.operation_time);
		builder.setView(root);
		builder.setPositiveButton(R.string.action_ok, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				int time;
				try {
					time = Integer.parseInt(operationTime.getText().toString());
				} catch (Exception e) {
					time = 0;
				}
				try {
					getReader().setOperationTime(time);
					MainActivity.mUserConfigs.setOperationTime(time);
					MainActivity.mUserConfigs.Save();
				} catch (AsReaderP252BException e) {
					AsReaderP252BLog.e(TAG, e, "ERROR. showOperationTimeDialog() - Failed to set operation time(%d)", time);
				}
				setOperationTime(time);
				InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(operationTime.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showOperationTimeDialog().$PositiveButton.onClick()");
				}
			}
		});
		builder.setNegativeButton(R.string.action_cancel, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(operationTime.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showOperationTimeDialog().$NegativeButton.onClick()");
				}
			}
		});
		builder.setCancelable(true);
		builder.setOnCancelListener(new DialogInterface.OnCancelListener() {

			@Override
			public void onCancel(DialogInterface dialog) {
				InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(operationTime.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showOperationTimeDialog().onCancel()");
				}
			}
		});
		AlertDialog dialog = builder.create();
		dialog.setOnShowListener(new DialogInterface.OnShowListener() {

			@Override
			public void onShow(DialogInterface dialog) {
				operationTime.setText(String.format(Locale.US, "%d", mOperationTime));
				operationTime.selectAll();
				operationTime.requestFocus();
				InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.showSoftInput(operationTime, InputMethodManager.SHOW_FORCED);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showOperationTimeDialog().onShow()");
				}
			}
		});
		dialog.show();

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showOperationTimeDialog()");
	}
}
