package support.andro.app.rfid.adapter;

import android.content.Context;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.asreader.app.rfid.R;
import com.asreader.p252b.diagnostics.AsReaderP252BLog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;

public class HumitureListAdapter extends BaseAdapter {
    private static final String TAG = TagListAdapter.class.getSimpleName();
    private static final int UPDATE_TIME = 300;

    private LayoutInflater mInflater;
    private ArrayList<HumitureListAdapter.HumitureItem> mList;
    private HashMap<String, HumitureListAdapter.HumitureItem> mMap;
    private boolean mIsDisplayPc;
    private boolean mIsReportRSSI;
    private Handler mHandler;
    private HumitureListAdapter.UpdateThread mUpdateThread;

    // ------------------------------------------------------------------------
    // Constructor
    // ------------------------------------------------------------------------
    public HumitureListAdapter(Context context) {
        super();

        mInflater = (LayoutInflater) context
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        mList = new ArrayList<>();
        mMap = new HashMap<>();
        mIsDisplayPc = true;
        mIsReportRSSI = false;
        mUpdateThread = null;
        mHandler = new Handler();
    }

    public void clear() {
        mList.clear();
        mMap.clear();
        notifyDataSetChanged();
    }

    @SuppressWarnings("unused")
    public boolean getDisplayPc() {
        return mIsDisplayPc;
    }

    public void setDisplayPc(boolean enabled) {
        mIsDisplayPc = enabled;
        notifyDataSetChanged();
    }

    @SuppressWarnings("unused")
    public boolean getReportRSSI() {
        return mIsReportRSSI;
    }

    public void setReportRSSI(boolean enabled) {
        if (mIsReportRSSI == enabled)
            return;
        mIsReportRSSI = enabled;
        clear();
    }

    public void start() {
        mUpdateThread = new HumitureListAdapter.UpdateThread();
        mUpdateThread.start();
    }

    public void shutDown() {
        if (mUpdateThread != null) {
            mUpdateThread.cancel();
            try {
                mUpdateThread.join();
            } catch (InterruptedException e) {
                AsReaderP252BLog.e(TAG, "ERROR. shutDown mUpdateThread.join() - [%s]", e.getMessage());
            }
            mUpdateThread = null;
        }
    }

    public void addTag(String tag, float rssi, float phase, float frequency, float humiture, boolean mIsTemperature) {
        HumitureListAdapter.HumitureItem item;

        if (null == (item = mMap.get(tag))) {
            item = new HumitureListAdapter.HumitureItem(tag, rssi, phase, frequency, humiture, mIsTemperature);
            mList.add(item);
            mMap.put(tag, item);
        } else {
            item.updateItem(rssi, phase, frequency, humiture, mIsTemperature);
        }
    }

    @Override
    public int getCount() {
        return mList.size();
    }

    @Override
    public String getItem(int position) {
        return mList.get(position).getTag(true);
    }

    public String getItem(int position, boolean displayPc) {
        return mList.get(position).getTag(displayPc);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        HumitureListAdapter.HumitureListViewHolder holder;

        if (null == convertView) {
            convertView = mInflater.inflate(R.layout.item_humiture_list, parent,
                    false);
            holder = new HumitureListAdapter.HumitureListViewHolder(convertView);
        } else {
            holder = (HumitureListAdapter.HumitureListViewHolder) convertView.getTag();
        }
        holder.setItem(mList.get(position), mIsDisplayPc);
        return convertView;
    }

    // ------------------------------------------------------------------------
    // Internal TagListItem class
    // ------------------------------------------------------------------------

    private class HumitureItem {

        private String mTag;
        private float mRSSI;
        private float mPhase;
        private float mFrequency;
        private float mHumiture;
        private boolean mIsTemperature;

        @SuppressWarnings("unused")
        public HumitureItem(String tag) {
            mTag = tag;
            mRSSI = 0;
            mPhase = 0;
            mFrequency = 0;
            mHumiture = 0;
            mIsDisplayPc = false;
        }

        private HumitureItem(String tag, float rssi, float phase, float frequency, float humiture , boolean isTemperature) {
            mTag = tag;
            mRSSI = rssi;
            mPhase = phase;
            mFrequency = frequency;
            mHumiture = humiture;
            mIsTemperature = isTemperature;
        }

        public String getTag(boolean displayPc) {
            return displayPc ? mTag : mTag.substring(4);
        }

        private float getRSSI() {
            return mRSSI;
        }

        public float getPhase() {
            return mPhase;
        }

        public float getFrequency() {
            return mFrequency;
        }

        public float getHumiture() {
            return mHumiture;
        }

        public boolean getIsTemperature() { return  mIsTemperature; }

        private void updateItem(float rssi, float phase, float frequency, float humiture, boolean isTemperature) {
            mRSSI = rssi;
            mPhase = phase;
            mFrequency = frequency;
            mHumiture = humiture;
            mIsTemperature = isTemperature;
        }
    }

    // ------------------------------------------------------------------------
    // Internal TagListViewHodler class
    // ------------------------------------------------------------------------
    private class HumitureListViewHolder {
        private TextView txtTag;
        private TextView txtHumitureValue;
        private TextView txtRssi;
        private TextView txtPhase;
        private TextView txtFrequency;
        private LinearLayout layoutSubItems;

        private float nSize = 13;
        private float nDecrease = (float) 0.3;

        private HumitureListViewHolder(View parent) {
            txtTag = (TextView) parent.findViewById(R.id.tag_value);
            txtHumitureValue = (TextView) parent.findViewById(R.id.humiture_value);
            txtRssi = (TextView)parent.findViewById(R.id.tag_rssi);
            txtPhase = (TextView)parent.findViewById(R.id.tag_phase);
            txtFrequency = (TextView)parent.findViewById(R.id.tag_frequency);
            layoutSubItems = (LinearLayout)parent.findViewById(R.id.sub_items);
            layoutSubItems.setVisibility(mIsReportRSSI ? View.VISIBLE : View.GONE);
            parent.setTag(this);
        }

        public void setItem(HumitureListAdapter.HumitureItem item, boolean displayPc) {
            txtTag.setText(item.getTag(displayPc));
            if (item.getIsTemperature()){
                txtHumitureValue.setText(String.format(Locale.US, "%.1f ", item.getHumiture()));
            }else  {
                txtHumitureValue.setText(String.format(Locale.US, "%.1f ", item.getHumiture()));
            }
            txtRssi.setText(String.format(Locale.US, "%.1f dB", item.getRSSI()));
            txtPhase.setText(String.format(Locale.US, "%.1f˚", item.getPhase()));
            txtFrequency.setText(String.format(Locale.US, "%.2fMHz", item.getFrequency()));
            layoutSubItems.setVisibility(mIsReportRSSI ? View.VISIBLE : View.GONE);
        }
    }

    // ------------------------------------------------------------------------
    // Internal UpdateThread class
    // ------------------------------------------------------------------------
    private class UpdateThread extends Thread {

        private boolean mIsAlived;

        private UpdateThread() {
            mIsAlived = false;
        }

        @Override
        public void run() {
            mIsAlived = true;

            while (mIsAlived) {
                synchronized (HumitureListAdapter.this) {
                    try {
                        HumitureListAdapter.this.wait(UPDATE_TIME);
                    } catch (InterruptedException e) {
                        AsReaderP252BLog.e(TAG, "ERROR. UpdateThread, TagListAdapter.this.wait(%d) - [%s]",
                                UPDATE_TIME, e.getMessage());
                    }
                }

                if (mIsAlived) {
                    mHandler.post(procUpdate);
                }
            }
        }

        private Runnable procUpdate = new Runnable() {

            @Override
            public void run() {
                HumitureListAdapter.this.notifyDataSetChanged();
            }

        };

        private void cancel() {
            synchronized (HumitureListAdapter.this) {
                mIsAlived = false;
                HumitureListAdapter.this.notify();
            }
        }
    }
}
