package support.andro.app.rfid.view.base;

import java.util.Locale;

import support.andro.app.rfid.adapter.SpinnerAdapter;
import support.andro.app.rfid.util.ResUtil;

import com.asreader.app.rfid.MainActivity;
import com.asreader.p252b.diagnostics.AsReaderP252BLog;
import com.asreader.p252b.rfid.exception.AsReaderP252BException;
import com.asreader.p252b.rfid.type.AsReaderP252BActionState;
import com.asreader.p252b.rfid.type.AsReaderP252BResultCode;
import com.asreader.app.rfid.R;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.ProgressBar;
import android.widget.TextView;

public abstract class AccessActivity extends RfidMaskActivity implements OnCheckedChangeListener {

	private static final String TAG = AccessActivity.class.getSimpleName();

	public static final int VIEW_READ_MEMORY = 1003;
	public static final int VIEW_WRITE_MEMORY = 1004;
	public static final int VIEW_TAG_ACCESS = 1005;

	private TextView txtSelection;
	private ProgressBar prgWait;
	private TextView txtMessage;
	private LinearLayout layoutMessage;
	private TextView txtPassword;
	private TextView txtPowerGain;
	private TextView txtOperationTime;
	private TextView txtReportRssi;
	private LinearLayout layoutRssi;
	private TextView txtRssi;
	private TextView txtPhase;
	private TextView txtFrequency;

	//private boolean mLastContinuousMode;
	private String mMask;
	private String mPassword;
	
	private boolean mReportRssi;

	private boolean mIsActionResult;

	public AccessActivity() {
		mMask = "";
		mIsActionResult = false;
	}

	@Override
	protected void initReader() {
		super.initReader();

		try {
			getReader().setContinuousMode(false);
		} catch (AsReaderP252BException e) {
			AsReaderP252BLog.e(TAG, e, "ERROR. initReader() - Failed to set continuous mode");
		}

		// Get RSSI Mode
		try {
			mReportRssi = getReader().getReportRSSI();
			runOnUiThread(new Runnable() {
				@Override
				public void run() {
					setVisibilityRssi(mReportRssi);
				}
			});
		} catch (Exception e) {
			AsReaderP252BLog.e(TAG, e, "ERROR. initReader() - Faield to get rssi report mode");
			layoutRssi.setVisibility(View.GONE);
		}
		displayReportRSSI();

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. initReader()");
	}

	@Override
	protected void destroyReader() {

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. destroyReader()");

//		try {
//			getReader().setContinuousMode(mLastContinuousMode);
//		} catch (AsReaderException e) {
//			ASRLog.e(TAG, e, "ERROR. destroyReader() - Failed to set continuous mode backup");
//		}

		super.destroyReader();
	}

	@Override
	public void onClick(View v) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onClick(%s)", ResUtil.getId(v.getId()));

		switch (v.getId()) {
		case R.id.password:
			showAccessPasswordDialog();
			break;
		case R.id.report_rssi:
			showReportRssiDialog();
			break;
		}

		super.onClick(v);
	}
	
	public void setVisibilityRssi(boolean visibility) {
		layoutRssi.setVisibility(visibility ? View.VISIBLE : View.GONE);
	}

	@Override
	public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onCheckedChanged(%s, %s)", ResUtil.getId(buttonView.getId()), isChecked);
	}

	// Stop Action
	protected void stopAction() {
		AsReaderP252BResultCode res;
		enableActionWidgets(false);
		outputMessage(R.string.stop_action_message);

		if ((res = getReader().stop()) != AsReaderP252BResultCode.NoError) {
			AsReaderP252BLog.e(TAG, "ERROR. stopAction() - Failed to stop operation [%s]", res);
			enableActionWidgets(true);
			outputFailMessage(R.string.stop_action_fail_message);
			return;
		}

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. stopAction()");
	}

	@Override
	protected void initWidgets() {
		super.initWidgets();

		// Selection TextView
		txtSelection = (TextView) findViewById(R.id.selection);
		// Wait ProgressBar
		prgWait = (ProgressBar) findViewById(R.id.progress_bar);
		// Message TextView
		txtMessage = (TextView) findViewById(R.id.message);
		// Message Background
		layoutMessage = (LinearLayout) findViewById(R.id.background);

		// RSSI Layout
		layoutRssi = (LinearLayout) findViewById(R.id.rssi_background);
		txtRssi = (TextView) findViewById(R.id.rssi);
		txtRssi.setText(R.string.default_rssi);
		txtPhase = (TextView) findViewById(R.id.phase);
		txtPhase.setText("0.0˚");
		txtFrequency = (TextView) findViewById(R.id.freq);
		txtFrequency.setText(R.string.default_frequency);

		// Password EditText
		txtPassword = (TextView) findViewById(R.id.password);
		txtPassword.setOnClickListener(this);
		setAccessPassword("");
		
		txtPowerGain = (TextView) findViewById(R.id.power_gain);
		txtPowerGain.setOnClickListener(this);
		txtOperationTime = (TextView) findViewById(R.id.operation_time);
		txtOperationTime.setOnClickListener(this);
		txtReportRssi = (TextView) findViewById(R.id.report_rssi);
		txtReportRssi.setOnClickListener(this);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. initWidgets()");
	}

	@Override
	protected void enableActionWidgets(boolean enabled) {
		super.enableActionWidgets(enabled);
		prgWait.setVisibility(getReader().getAction() == AsReaderP252BActionState.Stop ? View.GONE : View.VISIBLE);
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. enableActionWidgets(%s)", enabled);
	}

	@Override
	protected void clearWidgets() {
		outputSelection(" ");
		outputMessage(" ");
		txtRssi.setText(R.string.default_rssi);
		txtPhase.setText("0.0˚");
		txtFrequency.setText(R.string.default_frequency);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. clearWidgets()");
	}

	protected void setMask(String mask) {
		mMask = mask;
		txtSelection.setText(mask);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setMask([%s])", mask);
	}

	protected void outputSelection(String epc) {
		if (!mMask.equals(""))
			return;

		if(epc.length() > 36) {
			epc = epc.substring(0, 36);
			epc += "...";
		}

		txtSelection.setText(epc);
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. outputSelection([%s])", epc);
	}

	protected void outputMessage(int id) {
		outputMessage(getString(id));
	}

	protected void outputMessage(String msg) {
		txtMessage.setText(msg);
		txtMessage.setTextColor(getColor(getApplicationContext(), R.color.black));
		layoutMessage.setBackgroundColor(getColor(getApplicationContext(), R.color.message_background));

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. outputMessage([%s])", msg);
	}

	protected void outputSuccessMessage(int id) {
		outputSuccessMessage(getString(id));
	}

	protected void outputSuccessMessage(String msg) {
		prgWait.setVisibility(View.GONE);
		txtMessage.setText(msg);
		txtMessage.setTextColor(getColor(getApplicationContext(), R.color.black));
		layoutMessage.setBackgroundColor(getColor(getApplicationContext(), R.color.message_background));

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. outputSuccessMessage([%s])", msg);
	}

	protected void outputFailMessage(int id) {
		outputFailMessage(getString(id));
	}

	protected void outputFailMessage(String msg) {
		prgWait.setVisibility(View.GONE);
		txtMessage.setText(msg);
		txtMessage.setTextColor(getColor(getApplicationContext(), R.color.white));
		layoutMessage.setBackgroundColor(getColor(getApplicationContext(), R.color.red));

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. outputFailMessage([%s])", msg);
	}

	protected synchronized boolean isActionResult() {
		return mIsActionResult;
	}

	protected synchronized void setActionResult() {
		mIsActionResult = true;
	}

	protected synchronized void resetActionResult() {
		mIsActionResult = false;
	}

	protected void setAccessPassword(String password) {
		mPassword = password;
		txtPassword.setText(mPassword);
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setAccessPassword([%s])", password);
	}

	protected String getAccessPassword() {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. getAccessPassword() - [%s]", mPassword);
		return mPassword;
	}

	protected void setRssi(float rssi, float phase, float frequency) {
		txtRssi.setText(String.format(Locale.US, "%.1f dB", rssi));
		txtPhase.setText(String.format(Locale.US, "%.1f˚", phase));
		txtFrequency.setText(String.format(Locale.US, "%.2f MHz", frequency));

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setRssi(%.2f, %.2f, %.2f)", rssi, phase, frequency);
	}

	protected String getRequestCode(int requestCode) {
		switch (requestCode) {
		case VIEW_READ_MEMORY:
			return "VIEW_READ_MEMORY";
		case VIEW_WRITE_MEMORY:
			return "VIEW_WRITE_MEMORY";
		case VIEW_TAG_ACCESS:
			return "VIEW_TAG_ACCESS";
		}
		return super.getRequestCode(requestCode);
	}
	
	
	private void displayReportRSSI() {
		txtReportRssi.setText(mReportRssi ? "On" : "Off");
	}
	
	private void showReportRssiDialog() {
		if (!mIsEnabled)
			return;

		LinearLayout root = (LinearLayout) LinearLayout.inflate(this, R.layout.dialog_list_view, null);
		final ListView value = (ListView) root.findViewById(R.id.list);
		final SpinnerAdapter adapter = new SpinnerAdapter(this, android.R.layout.simple_list_item_single_choice);
		adapter.addItem(0, "Off");
		adapter.addItem(1, "On");
		value.setAdapter(adapter);
		value.setChoiceMode(ListView.CHOICE_MODE_SINGLE);

		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setTitle(R.string.report_rssi);
		builder.setView(root);
		builder.setPositiveButton(R.string.action_ok, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				try {
					int position = value.getCheckedItemPosition();
					mReportRssi = adapter.getValue(position) > 0;
					getReader().setReportRSSI(mReportRssi);
					MainActivity.mUserConfigs.setReportRssi(mReportRssi);
					MainActivity.mUserConfigs.Save();
					setVisibilityRssi(mReportRssi);
				} catch (Exception e) {
				}
				displayReportRSSI();
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAlgorithmDialog().$PositiveButton.onClick()");
			}
		});
		builder.setNegativeButton(R.string.action_cancel, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAlgorithmDialog().$NegativeButton.onClick()");
			}
		});
		builder.setCancelable(true);
		builder.setOnCancelListener(new DialogInterface.OnCancelListener() {

			@Override
			public void onCancel(DialogInterface dialog) {
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAlgorithmDialog().onCancel()");
			}
		});
		AlertDialog dialog = builder.create();
		dialog.setOnShowListener(new DialogInterface.OnShowListener() {

			@Override
			public void onShow(DialogInterface dialog) {
				int position = adapter.getPosition(mReportRssi ? 1 : 0);
				value.setItemChecked(position, true);
				value.setSelectionFromTop(position, 0);
				AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAlgorithmDialog().onShow()");
			}
		});
		dialog.show();
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAlgorithmDialog()");
	}

	private void showAccessPasswordDialog() {
		if (!mIsEnabled)
			return;
		
		LinearLayout root = (LinearLayout)LinearLayout.inflate(this, R.layout.dialog_password, null);
		final EditText password = (EditText)root.findViewById(R.id.password);
		
		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setTitle(R.string.password);
		builder.setView(root);
		builder.setPositiveButton(R.string.action_ok, new DialogInterface.OnClickListener() {
			
			@Override
			public void onClick(DialogInterface dialog, int which) {
				setAccessPassword(password.getText().toString());
				InputMethodManager imm = (InputMethodManager)getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(password.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAccessPasswordDialog().$PositiveButton.onClick()");
				}
			}
		});
		builder.setNegativeButton(R.string.action_cancel, new DialogInterface.OnClickListener() {
			
			@Override
			public void onClick(DialogInterface dialog, int which) {
				InputMethodManager imm = (InputMethodManager)getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(password.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAccessPasswordDialog().$NegativeButton.onClick()");
				}
			}
		});
		builder.setCancelable(true);
		builder.setOnCancelListener(new DialogInterface.OnCancelListener() {
			
			@Override
			public void onCancel(DialogInterface dialog) {
				InputMethodManager imm = (InputMethodManager)getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.hideSoftInputFromWindow(password.getWindowToken(), 0);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAccessPasswordDialog().onCancel()");
				}
			}
		});
		AlertDialog dialog = builder.create();
		dialog.setOnShowListener(new DialogInterface.OnShowListener() {
			
			@Override
			public void onShow(DialogInterface dialog) {
				password.setText(mPassword);
				password.selectAll();
				password.requestFocus();
				InputMethodManager imm = (InputMethodManager)getSystemService(Context.INPUT_METHOD_SERVICE);
				if(imm != null) {
					imm.showSoftInput(password, InputMethodManager.SHOW_FORCED);
					AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAccessPasswordDialog().onShow()");
				}
			}
		});
		dialog.show();
		
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. showAccessPasswordDialog()");
	}
}
