package support.andro.app.rfid.adapter;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import com.asreader.app.rfid.R;
import com.asreader.p252b.barcode.AsReaderP252BBarcodeType;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.TextView;

public class BarcodeListAdapter extends BaseAdapter {

	private LayoutInflater mInflater;
	private ArrayList<BarcodeListItem> mList;
	private HashMap<String, BarcodeListItem> mMap;

	public BarcodeListAdapter(Context context) {
		super();
		
		this.mInflater = (LayoutInflater) context
				.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		this.mList = new ArrayList<>();
		this.mMap = new HashMap<>();
	}

	public void clear() {
		this.mList.clear();
		this.notifyDataSetChanged();
		this.mMap.clear();
	}
	
	public void addItem(AsReaderP252BBarcodeType type, String barcode) {
		BarcodeListItem item;
		String tag = type + barcode;
		if (null == (item = mMap.get(tag))) {
			item = new BarcodeListItem(type, barcode);
			mList.add(item);
			mMap.put(tag, item);
		} else {
			item.updateItem();
		}
		
		notifyDataSetChanged();
	}
	
	@Override
	public int getCount() {
		return this.mList.size();
	}

	@Override
	public String getItem(int position) {
		return this.mList.get(position).getBarcode();
	}

	@Override
	public long getItemId(int position) {
		return position;
	}
	
	public String getItemForCsv(int position) {
		BarcodeListItem item = mList.get(position);
		return String.format("%s,%s", item.getBarcode(), item.getTime());
	}

	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		BarcodeListViewHolder holder;
		
		if (null == convertView) {
			convertView = this.mInflater.inflate(R.layout.item_barcode_list, parent, false);
			holder = new BarcodeListViewHolder(convertView);
		} else {
			holder = (BarcodeListViewHolder)convertView.getTag();
		}
		holder.setItem(this.mList.get(position));
		return convertView;
	}

	private class BarcodeListItem {
		private AsReaderP252BBarcodeType mType;
		private String mBarcode;
		private int mCount;
		private long mTime;

		private SimpleDateFormat mDateFormat;

		private BarcodeListItem(AsReaderP252BBarcodeType type, String barcode) {
			mType = type;
			mBarcode = barcode;
			mCount = 1;
			mDateFormat = new SimpleDateFormat("yyy/MM/dd HH:mm:ss", Locale.US);
		}

		public AsReaderP252BBarcodeType getType() {
			return mType;
		}

		public String getBarcode() {
			return mBarcode;
		}
		
		public int getCount() {
			return mCount;
		}
		
		private void updateItem() {
			mCount++;
		}
		
		public String getTime() {
			return mDateFormat.format(new Date(mTime));
		}
	}

	private class BarcodeListViewHolder {
		private TextView txtType;
		private TextView txtBarcode;
		private TextView txtCount;

		private BarcodeListViewHolder(View parent) {
			txtType = (TextView) parent.findViewById(R.id.code_type);
			txtBarcode = (TextView) parent.findViewById(R.id.barcode);
			txtCount = (TextView) parent.findViewById(R.id.barcode_count);
			parent.setTag(this);
		}

		public void setItem(BarcodeListItem item) {
			txtType.setText(item.getType().toString());
			txtBarcode.setText(item.getBarcode());
			txtCount.setText(String.format(Locale.US, "%d", item.getCount()));
		}
	}
}
