package com.asreader.app.rfid;

import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.bluetooth.BluetoothDevice;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.graphics.Color;
import android.Manifest;
import android.os.Bundle;
import android.os.Build;
import android.os.Handler;
import android.os.PowerManager;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import android.util.Log;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnTouchListener;
import android.view.Window;
import android.widget.Button;
import android.widget.ImageView;

import android.widget.PopupWindow;
import android.widget.TextView;
import android.widget.Toast;

import com.asreader.app.rfid.interfaces.OnSearchListener;

import com.asreader.app.rfid.view.BarcodeDemoActivity;
import com.asreader.app.rfid.view.HumitureActivity;
import com.asreader.app.rfid.view.InventoryActivity;
import com.asreader.app.rfid.view.OptionActivity;
import com.asreader.app.rfid.view.ReadMemoryActivity;

import com.asreader.app.rfid.view.TagAccessActivity;
import com.asreader.app.rfid.view.UpdateFirmwareActivity;
import com.asreader.app.rfid.view.RFIDUpdateFirmwareActivity;
import com.asreader.app.rfid.view.WriteMemoryActivity;

import support.andro.app.rfid.dialog.WaitDialog;

import support.andro.app.rfid.util.AsReaderP252BUserConfigs;
import support.andro.app.rfid.util.UserConfigs;
import support.andro.app.rfid.view.base.AccessActivity;
import support.andro.app.rfid.view.base.RfidActivity;

import com.asreader.app.rfid.view.widges.ShowSearchWindow;
import com.asreader.p252b.AsReaderP252B;

import com.asreader.p252b.barcode.AsReaderP252BBarcodeType;
import com.asreader.p252b.device.AsReaderP252BDevice;
import com.asreader.p252b.device.AsReaderP252BDeviceBluetoothCdc;
import com.asreader.p252b.device.AsReaderP252BDeviceUsbCdc;
import com.asreader.p252b.device.bluetoothEvent.AsReaderP252BBluetoothDiscoveryEventListener;
import com.asreader.p252b.device.type.AsReaderP252BConnectionState;
import com.asreader.p252b.device.type.AsReaderP252BKeyState;
import com.asreader.p252b.diagnostics.AsReaderP252BLog;
import com.asreader.p252b.rfid.event.AsReaderP252BEventListener;
import com.asreader.p252b.rfid.exception.AsReaderP252BException;
import com.asreader.p252b.rfid.type.AsReaderP252BActionState;
import com.asreader.p252b.rfid.type.AsReaderP252BResultCode;

import com.asreader.p252b.AsReaderP252BManager;


public class MainActivity extends Activity implements OnClickListener, OnTouchListener, AsReaderP252BEventListener {

	private static final String TAG = MainActivity.class.getSimpleName();

	private static boolean WRITE_LOG = true;
	private static final String LOG_PATH = "Log";
	private static final String LOG_PREFIX = "AsReaderP252BDemo";

	public static final String APP_NAME = "come.asreader.p252b";
	private static final String KEY_BATTERY_INTERVAL = "battery_interval";
	private static final String KEY_MASK_TYPE = "mask_type";
	private static final int DEFAULT_BATTERY_INTERVAL = 10;
	private static final int DEFAULT_MASK_TYPE = 0;
	private static final int VIEW_INVENTORY = 1000;
	private static final int VIEW_OPTION = 1006;
	private static final int VIEW_BARCODE_DEMO = 1007;
	private static final int VIEW_BARCODE_OPTION = 1008;
	private static final int VIEW_HUMITURE = 1009;
	private static final int RESULT_DISCONNECTED = Activity.RESULT_FIRST_USER + 1;

	private static final int REQUEST_PERMISSION_CONTACTS = 1000;

	private PowerManager.WakeLock wakeLock = null;
	private AsReaderP252B mReader = null;

	private TextView txtDemoVersion;
	private TextView txtFirmwareVersion;
	private TextView txtHardwareVersion;
	private TextView txtAddress;
	private TextView txtGlobalBand;
	private TextView txtSerialNumber;
	private Button btnInventory;
	private Button btnReadMemory;
	private Button btnWriteMemory;
	private Button btnTagAccess;
	private Button btnOption;
	private Button btnUpdateFirmware;
	private Button btnRFIDUpdateFirmware;
	private Button btnHumiture;
	private Button btnBarcodeScan;
	private ImageView imgBattery;
	private ImageView imgLogo;

	//private String mDeviceAddress;
	private int mBatteryInterval;
	private boolean mKeyAction;

	private Timer timerCheckBattery;
	private Timer timerCheckConnect;

	private Handler mHandler;
	private AsReaderP252BDevice mAsReaderP252BDevice = null;

	private final int COLOR_ENABLED = Color.BLACK;
	private final int COLOR_DISABLED = Color.LTGRAY;

	private String[] mPermissions = null;
	public static UserConfigs mUserConfigs = null;

	private MenuItem mnuConnectUsbDevice;
	private MenuItem mnuDisconnectDevice;
	private MenuItem mnuConnectBluetoothDevice;
	private MenuItem mnuBluetoothDisconnect;

	private boolean isFinishedDiscovery;
	private boolean isBleDevConecting;
	private ShowSearchWindow showSearchWindow;

	// Bluetooth
	private static AsReaderP252BDeviceBluetoothCdc deviceBluetoothCdc;

	//USB
	private static AsReaderP252BDeviceUsbCdc usbCdc;

	private AsReaderP252BUserConfigs asReaderP252BUserConfigs;
	@SuppressWarnings("deprecation")
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		deviceBluetoothCdc = new AsReaderP252BDeviceBluetoothCdc(this);
		usbCdc = new AsReaderP252BDeviceUsbCdc(this);

		requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS);
		setProgressBarIndeterminateVisibility(true);
		setContentView(R.layout.activity_main);

		getBluetoothLocationPermission();

		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
			mPermissions = new String[]{Manifest.permission.WAKE_LOCK, Manifest.permission.READ_MEDIA_IMAGES, Manifest.permission.READ_MEDIA_VIDEO,
					Manifest.permission.BLUETOOTH_ADMIN, Manifest.permission.BLUETOOTH_SCAN, Manifest.permission.BLUETOOTH_CONNECT};
		} else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
			mPermissions = new String[]{Manifest.permission.WAKE_LOCK, Manifest.permission.WRITE_EXTERNAL_STORAGE,
					Manifest.permission.READ_EXTERNAL_STORAGE,Manifest.permission.BLUETOOTH_ADMIN,Manifest.permission.BLUETOOTH_SCAN,Manifest.permission.BLUETOOTH_CONNECT};
		} else {
			mPermissions = new String[]{Manifest.permission.WAKE_LOCK, Manifest.permission.WRITE_EXTERNAL_STORAGE,
					Manifest.permission.READ_EXTERNAL_STORAGE,Manifest.permission.BLUETOOTH_ADMIN,Manifest.permission.BLUETOOTH_CONNECT};
		}

		mUserConfigs = new UserConfigs(this.getApplicationContext());
		asReaderP252BUserConfigs = AsReaderP252BUserConfigs.getInstance();
		asReaderP252BUserConfigs.setContext(this.getApplicationContext());
		// Initialize Widget
		initWidgets();

		// Disable butotns
		enableMenuButtons(false);

		checkPermission();
		connectLastDevice();
	}

	private void connectLastDevice() {
		if (asReaderP252BUserConfigs.getConnectType() == AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_Bluetooth) {
			SharedPreferences ps = getSharedPreferences("BluetoothDevice", Context.MODE_PRIVATE);
			String deviceAddress = ps.getString("address", "");
			if (!deviceAddress.equals("")) {
				if (mReader != null){
					isBleDevConecting = true;
					mAsReaderP252BDevice = deviceBluetoothCdc;
					mAsReaderP252BDevice.setAddress(deviceAddress);
					showWaitDialog();
					mReader.connectDevice(mAsReaderP252BDevice);
					asReaderP252BUserConfigs.setConnectType(AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_Bluetooth);
					checkPermission();
				}
			}
		}
	}

	private boolean checkPermission() {
		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
			onPermissionsResult(true);
			return false;
		}

		ArrayList<String> permissions = new ArrayList<String>();
		int grantResult = PackageManager.PERMISSION_GRANTED;

		for (int i = 0; i < mPermissions.length; i++) {
			grantResult = PackageManager.PERMISSION_GRANTED;
			grantResult = ContextCompat.checkSelfPermission(this, mPermissions[i]);
			if (grantResult == PackageManager.PERMISSION_DENIED) {
				permissions.add(mPermissions[i]);
			}
		}
		if (permissions.size() <= 0) {
			onPermissionsResult(true);
			return false;
		}

		String[] requestPermissions = new String[permissions.size()];
		permissions.toArray(requestPermissions);
		ActivityCompat.requestPermissions(this, permissions.toArray(requestPermissions), REQUEST_PERMISSION_CONTACTS);

		AsReaderP252BLog.e(TAG, "INFO. checkPermission()");
		return true;
	}

	private final int ACCESS_LOCATION = 1;

	@SuppressLint("WrongConstant")
	private void getBluetoothLocationPermission() {
		if (Build.VERSION.SDK_INT > Build.VERSION_CODES.M) {
			int permissionCheck = 0;
			permissionCheck = this.checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION);
			permissionCheck += this.checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION);
			permissionCheck += this.checkSelfPermission(Manifest.permission.BLUETOOTH_ADMIN);
			permissionCheck += this.checkSelfPermission(Manifest.permission.BLUETOOTH_SCAN);
			permissionCheck += this.checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT);

			if (permissionCheck != PackageManager.PERMISSION_GRANTED) {
				this.requestPermissions(
						new String[]{Manifest.permission.ACCESS_FINE_LOCATION,
								Manifest.permission.ACCESS_COARSE_LOCATION,
								Manifest.permission.BLUETOOTH_ADMIN,
								Manifest.permission.BLUETOOTH_SCAN,
								Manifest.permission.BLUETOOTH_CONNECT},
						ACCESS_LOCATION);
			}
		}
	}

	@Override
	public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
		if (requestCode != REQUEST_PERMISSION_CONTACTS)
			return;

		boolean result = true;
		for (int i = 0; i < grantResults.length; i++) {
			result &= grantResults[i] != PackageManager.PERMISSION_DENIED;
		}

		onPermissionsResult(result);
	}

	private void onPermissionsResult(boolean result) {
		if (!result) {
			AsReaderP252BLog.e(TAG, "Failed to get permissions");
			setProgressBarIndeterminateVisibility(false);
			return;
		}

		if (WRITE_LOG)
			AsReaderP252BLog.startUp(LOG_PATH, LOG_PREFIX);

		// Using higher levels can cause serious performance degradation.
		AsReaderP252BLog.setLogLevel(AsReaderP252BLog.ERROR);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onCreate()");

		// Setup Always Wakeup
		setupAlwaysWakeup();

		// Load Configuration
		loadConfig();

		// Initialize RFID Reader
		initAsReader();

		// Display Demo Version
		displayVersion();

		setProgressBarIndeterminateVisibility(false);
	}

	@Override
	protected void onPause() {
		if (mReader != null) {
			AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onPause()");

			saveConfig();
			stopCheckBattery();
		}
		super.onPause();
	}

	@Override
	protected void onResume() {
		if (mReader != null) {
			AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onResume()");

			if (mReader.getState() == AsReaderP252BConnectionState.Connected) {
				startCheckBattery();
				mReader.setEventListener(this);
			} else {
				setDisconnectedState();
			}
		}
		super.onResume();
	}

	@Override
	protected void onStop() {
		if (mReader != null) {
			AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onStop()");
		}
		super.onStop();
	}

	@Override
	protected void onDestroy() {
		if (mReader != null) {
			AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onDestroy()");

			saveConfig();
			if (timerCheckBattery != null) {
				timerCheckBattery.cancel();
				timerCheckBattery = null;
			}
			if (timerCheckConnect != null) {
				timerCheckConnect.cancel();
				timerCheckConnect = null;
			}
			// Destroy Rfid Reader
			AsReaderP252BManager.onDestroy();

			// Destroy USB device
			if (mAsReaderP252BDevice != null) {
				mAsReaderP252BDevice.destroy();
				mAsReaderP252BDevice = null;
			}

			defaultWakeup();

			AsReaderP252BLog.shutdown();
		}
		super.onDestroy();
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {

		AsReaderP252BLog.d(TAG, AsReaderP252BLog.INFO, "DEBUG. onActivityResult(%s, %s)", getRequestCode(requestCode), getResultCode(resultCode));

		switch (requestCode) {
			case VIEW_INVENTORY:
			case AccessActivity.VIEW_READ_MEMORY:
			case AccessActivity.VIEW_WRITE_MEMORY:
			case AccessActivity.VIEW_TAG_ACCESS:
			case VIEW_BARCODE_DEMO:
			case VIEW_BARCODE_OPTION:
				switch (resultCode) {
					case RESULT_DISCONNECTED:
						setDisconnectedState();
						break;
					default:
						break;
				}
				break;
			case VIEW_OPTION:
				switch (resultCode) {
					case RESULT_DISCONNECTED:
						setDisconnectedState();
						break;
					case RESULT_OK:
						if (data != null) {
							mBatteryInterval = data.getIntExtra(KEY_BATTERY_INTERVAL, DEFAULT_BATTERY_INTERVAL);
						}
						break;
				}
				break;
		}

		super.onActivityResult(requestCode, resultCode, data);
	}

	@Override
	public void onReadBarcode(AsReaderP252B reader, AsReaderP252BBarcodeType type, String barcode) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onReadBarcode (%s, [%s])", type, barcode);
	}

	@Override
	public void onReadBarcode(AsReaderP252B reader, AsReaderP252BBarcodeType type, byte[] barcodeData) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onReadBarcode (%s, [%s])", type, barcodeData);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onCreateOptionsMenu()");
		getMenuInflater().inflate(R.menu.main, menu);
		return true;
	}

	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {

		if(mReader != null) {
			AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onPrepareOptionsMenu()");
			mnuConnectUsbDevice = menu.findItem(R.id.menu_connect_device_usb);
			mnuConnectBluetoothDevice = menu.findItem(R.id.menu_search_bluetooth);
			mnuDisconnectDevice = menu.findItem(R.id.menu_disconnect);
			mnuBluetoothDisconnect = menu.findItem(R.id.menu_bluetooth_disconnect);
			visibleMenuItem(mReader.getState());
		}
		return super.onPrepareOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		int id = item.getItemId();
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onOptionsItemSelected(%s)", getMenuId(id));

		switch (id) {
			case R.id.menu_search_bluetooth:
				showSearchPopWindow(txtFirmwareVersion);
				return true;
			case R.id.menu_connect_device_usb:
				mAsReaderP252BDevice = AsReaderP252BManager.getInstance().getCurrentDevice();
				if (mAsReaderP252BDevice == null) {
					mAsReaderP252BDevice = usbCdc;
					showWaitDialog();
					if (mReader != null) {
						mReader.connectDevice(mAsReaderP252BDevice);
					}
					asReaderP252BUserConfigs.setConnectType(AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_USB);
					SharedPreferences sp = getSharedPreferences("USBDevice", Context.MODE_PRIVATE);
					SharedPreferences.Editor edit = sp.edit();
					edit.clear();
					edit.putString("usb", "1");
					edit.commit();
				} else {
					onStateChanged(mReader, AsReaderP252BConnectionState.Connected);
				}
				checkPermission();
				return true;

			case R.id.menu_disconnect:
				stopCheckBattery();
				WaitDialog.show(this, R.string.disconnecting_reader);
				mReader.disconnectDevice();
				SharedPreferences sp = getSharedPreferences("USBDevice", Context.MODE_PRIVATE);
				SharedPreferences.Editor edit = sp.edit();
				edit.clear();
				edit.putString("usb", "");
				edit.commit();
				return true;
			case R.id.menu_bluetooth_disconnect:
				// Disconnect Device
				stopCheckBattery();
				WaitDialog.show(this, R.string.disconnecting_reader);
				mReader.disconnectDevice();
				SharedPreferences spp = getSharedPreferences("BluetoothDevice", Context.MODE_PRIVATE);
				SharedPreferences.Editor editt = spp.edit();
				editt.clear();
				editt.putString("address", "");
				editt.commit();
				return true;
		}
		return super.onOptionsItemSelected(item);
	}

	private Runnable mActionCheckConnect = new Runnable() {
		@Override
		public void run() {
			if (timerCheckConnect != null) {
				enableMenuButtons(false);
				visibleMenuItem(AsReaderP252BConnectionState.Disconnected);
				WaitDialog.hide();
			}
		}
	};
	@Override
	public void onClick(View v) {
		Intent intent;
		int id = v.getId();

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. onClick(%s)", getOnClickId(id));

		switch (id) {
			case R.id.inventory:
				intent = new Intent(this, InventoryActivity.class);
				startActivityForResult(intent, VIEW_INVENTORY);
				break;
			case R.id.read_memory:
				intent = new Intent(this, ReadMemoryActivity.class);
				startActivityForResult(intent, AccessActivity.VIEW_READ_MEMORY);
				break;
			case R.id.write_memory:
				intent = new Intent(this, WriteMemoryActivity.class);
				startActivityForResult(intent, AccessActivity.VIEW_WRITE_MEMORY);
				break;
			case R.id.tag_access:
				intent = new Intent(this, TagAccessActivity.class);
				startActivityForResult(intent, AccessActivity.VIEW_TAG_ACCESS);
				break;
			case R.id.option:
				intent = new Intent(this, OptionActivity.class);
				intent.putExtra(KEY_BATTERY_INTERVAL, mBatteryInterval);
				intent.putExtra(RfidActivity.KEY_ACTION, mKeyAction);
				startActivityForResult(intent, VIEW_OPTION);
				break;
			case R.id.humiture:
				intent = new Intent(this, HumitureActivity.class);
				startActivityForResult(intent, VIEW_HUMITURE);
				break;
			case R.id.update_firmware:
				intent = new Intent(this, UpdateFirmwareActivity.class);
				startActivity(intent);
				break;
			case R.id.rfid_update_firmware:
				intent = new Intent(this, RFIDUpdateFirmwareActivity.class);
				startActivity(intent);
				break;
			case R.id.barcode_scan:
				intent = new Intent(this, BarcodeDemoActivity.class);
				startActivity(intent);
			default:
				break;
		}
	}

	private void showSearchPopWindow(View view) {
		deviceBluetoothCdc.setBluetoothDiscoveryEventListener(mAsReaderP252BBluetoothDiscoveryEventListener);
		isBleDevConecting = false;
		isFinishedDiscovery = true;
		showSearchWindow = new ShowSearchWindow(this, new OnSearchListener() {
			@Override
			public void onStartDiscovery() {
				Log.i(TAG, "===onStartDiscovery" );
				if (isFinishedDiscovery) {
					setProgressBarIndeterminateVisibility(true);
					showSearchWindow.getmNewDevicesAdapter().clear();
					deviceBluetoothCdc.startDiscovery();
					showSearchWindow.refreshTextView(true);
					isFinishedDiscovery = false;
				} else {
					setProgressBarIndeterminateVisibility(false);
					deviceBluetoothCdc.stopDiscovery();
				}
			}

			@Override
			public void onCancelDiscovery() {
				setProgressBarIndeterminateVisibility(false);
				deviceBluetoothCdc.stopDiscovery();
			}

			@Override
			public void onConnect(String address) {
				if (mReader != null){
					isBleDevConecting = true;
					mAsReaderP252BDevice = deviceBluetoothCdc;
					mAsReaderP252BDevice.setAddress(address);
					SharedPreferences sp = getSharedPreferences("BluetoothDevice", Context.MODE_PRIVATE);
					SharedPreferences.Editor edit = sp.edit();
					edit.clear();
					edit.putString("address", address);
					edit.commit();
					showWaitDialog();
					mReader.connectDevice(mAsReaderP252BDevice);
					asReaderP252BUserConfigs.setConnectType(AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_Bluetooth);
					checkPermission();
				}else {
					setProgressBarIndeterminateVisibility(false);
				}
			}
		});
		showSearchWindow.showAtBottomWithAnima(view);
		showSearchWindow.setOnDismissListener(new PopupWindow.OnDismissListener() {
			@Override
			public void onDismiss() {
				if (deviceBluetoothCdc != null) {
					deviceBluetoothCdc.stopDiscovery();
				}
				if (!isBleDevConecting) {
					setProgressBarIndeterminateVisibility(false);
				}

			}
		});
	}

	private void showWaitDialog() {
		if (timerCheckConnect != null) {
			timerCheckConnect.cancel();
			timerCheckConnect = null;
		}
		timerCheckConnect = new Timer();
		timerCheckConnect.schedule(new TimerTask() {

			@Override
			public void run() {
				runOnUiThread(mActionCheckConnect);
			}

		}, 20000);
		WaitDialog.show(this, R.string.connecting_reader);
	}
	private AsReaderP252BBluetoothDiscoveryEventListener mAsReaderP252BBluetoothDiscoveryEventListener = new AsReaderP252BBluetoothDiscoveryEventListener() {
		@Override
		public void onReceivedDevice(final BluetoothDevice device) {
			if (showSearchWindow != null && showSearchWindow.isShowing()) {
				runOnUiThread(new Runnable() {
					@Override
					public void run() {
						if (showSearchWindow.getmNewDevicesAdapter().getPosition(device.getName() + "\n" + device.getAddress()) == -1
								&& device.getName() != null && (device.getName().contains(getResources().getString(R.string.device_name)) || device.getName().contains(getResources().getString(R.string.device_nameP23)) || device.getName().contains(getResources().getString(R.string.device_nameW3)) || device.getName().contains(getResources().getString(R.string.device_namePaddleType)))) {
							showSearchWindow.getmNewDevicesAdapter().add(device.getName() + "\n" + device.getAddress());
							isFinishedDiscovery = false;

						}
					}
				});
			}

		}

		@Override
		public void onFoundDeviceFinished() {
			if (showSearchWindow != null && showSearchWindow.isShowing()) {
				runOnUiThread(new Runnable() {
					@Override
					public void run() {
						isFinishedDiscovery = true;
						showSearchWindow.refreshTextView(false);
						setProgressBarIndeterminateVisibility(false);
					}
				});
			}

		}
	};

	@SuppressWarnings("deprecation")
	@Override
	public void onStateChanged(AsReaderP252B reader, final AsReaderP252BConnectionState state) {

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onConnectionStateChanged(%s)", state);

		switch (state) {
		case Disconnected:
			if (mReader.getResultCode() == AsReaderP252BResultCode.NotSupportFirmware) {
				AlertDialog.Builder alert = new AlertDialog.Builder(this);
				alert.setTitle(R.string.system_alert);
				alert.setIcon(android.R.drawable.ic_dialog_alert);
				alert.setMessage(R.string.not_support_firmware);
				alert.setPositiveButton(R.string.action_ok, null);
				alert.show();
			}
			setDisconnectedState();
			setProgressBarIndeterminateVisibility(false);
			break;
		case Connecting:
			setProgressBarIndeterminateVisibility(true);
		case Listen:
			enableMenuButtons(false);
			imgLogo.setImageResource(R.drawable.asreader_log_off);
			break;
		case Connected:
			if (timerCheckConnect != null) {
				timerCheckConnect.cancel();
				timerCheckConnect = null;
			}
			new Thread(new Runnable() {
				@Override
				public void run() {
					String versionName = "";
					String hardVersion = "";
					String serialNumber = "";
					String rfModuleVersion = "";
					String globalBand = "";
					try {
						versionName = mReader.getFirmwareVersion();
						hardVersion = mReader.getHardwareVersion();
						serialNumber = mReader.getSerialNumber();
						rfModuleVersion = mReader.getRFModuleVersion();
						globalBand = mReader.getRegion();
					} catch(Exception ae) {
						AsReaderP252BLog.e(TAG, "ERROR. Connected Thread - exit by exception.");
						return;
					}

					if( mReader.getState() != AsReaderP252BConnectionState.Connected) {
						AsReaderP252BLog.e(TAG, "ERROR. Connected Thread - exit thread. state is not connected.");
						return;
					}
					if(!mUserConfigs.Load(mReader)) {
						AsReaderP252BLog.e(TAG, "ERROR. Connected Thread - failed to load mask data.");
						return;
					}
					try {
						mReader.setQuerySession(mUserConfigs.getSession());
						mReader.setSessionFlag(mUserConfigs.getTarget());
						mReader.setReportRSSI(mUserConfigs.getReportRssi());
						mReader.setPowerGain(mUserConfigs.getPowerGain());
						mReader.setContinuousMode(mUserConfigs.getIsContinuousMode());
						mReader.setOperationTime(mUserConfigs.getOperationTime());
						mReader.setBuzzer(mUserConfigs.getBuzzerState());
						mReader.setIdleTime(mUserConfigs.getIdleTime());
						mReader.setSleepTime(mUserConfigs.getSleepTime());
						mReader.setAutoOffTime(mUserConfigs.getAutoOffTime());
						mReader.setQValue(mUserConfigs.getQValue());
						mReader.setLinkProfile(mUserConfigs.getDefaultLinkProfileValue());
						mReader.setBarcodeTimeOut(mUserConfigs.getBarcodeTimeout());
//						boolean sta = mUserConfigs.getIsFrequencyAutomatic();
//						mReader.setFrequencyAutomatic(sta);
//						if (!sta)
//						{
//							LbtItem[] mFreqTable = mReader.getFrequency();
//							if (mFreqTable != null) {
//								String frequency = mUserConfigs.getFrequency();
//								if (frequency.equals(""))
//								{
//									return;
//								}
//								ArrayList<String> mList = new ArrayList<>();
//								String[] result = frequency.split(",");
//								for(int i = 0; i < result.length; i++)
//								{
//									mList.add(result[i]);
//								}
//								for (int i = 0; i < mFreqTable.length; i++) {
//									LbtItem item = mFreqTable[i];
//									if (mList.contains(String.valueOf(i))) {
//										item.setUsed(true);
//									}else{
//										item.setUsed(false);
//									}
//								}
//								mReader.setFrequency(mFreqTable);
//							}
//						}
						try {
							mReader.setBaudRate(mUserConfigs.getBaudRate());
						} catch (Exception ae) {
							AsReaderP252BLog.e(TAG, "ERROR. setBaudRate");
						}
					} catch (Exception ae) {
						AsReaderP252BLog.e(TAG, "ERROR. Connected Thread - exit by exception.");
						return;
					}
					saveConfig();
					runOnUiThread(new InitRunnable(versionName,hardVersion,serialNumber, rfModuleVersion, globalBand));
				}
			}).start();
			break;
		default:
			break;
			
		}
	}

	@Override
	public void onActionChanged(AsReaderP252B reader, AsReaderP252BActionState action) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onActionChanged(%s)", action);
	}

	@Override
	public void onReadTag(AsReaderP252B reader, AsReaderP252BActionState action, String tag, float rssi, float phase, float frequency) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onReadTag(%s, [%s], %.2f, %.2f)", action, tag, rssi, phase);
	}

	@Override
	public void onAccessResult(AsReaderP252B reader, AsReaderP252BResultCode code, AsReaderP252BActionState action, String epc, String data,
							   float rssi, float phase, float frequency) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onAccessResult(%s, %s, [%s], [%s], %.2f, %.2f)", code, action, epc, data, rssi, phase);
	}

	@Override
	public void onReadTemperatureTag(AsReaderP252B reader, AsReaderP252BActionState action, String tag, float rssi, float phase, float frequency, float temperature) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onReadTag(%s, [%s], %.2f, %.2f, %.2f, %.2f)", action, tag, rssi, phase, frequency, temperature);
	}

	@Override
	public void onReadHumidityTag(AsReaderP252B reader, AsReaderP252BActionState action, String tag, float rssi, float phase, float frequency, float humidity) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onReadTag(%s, [%s], %.2f, %.2f, %.2f, %.2f)", action, tag, rssi, phase, frequency, humidity);
	}

	@Override
	public void onReceivedData(byte[] data) {

	}

	@Override
	public boolean onKeyEvent(AsReaderP252B reader, AsReaderP252BKeyState state) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onKeyEvent [%s]", state);
		return false;
	}

	@Override
	public boolean onModeKeyEvent(AsReaderP252B reader, AsReaderP252BKeyState state) {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "EVENT. onModeKeyEvent [%s]", state);
		return false;
	}

	// Initialize Widgets
	private void initWidgets() {

		txtDemoVersion = (TextView) findViewById(R.id.demo_version);
		txtFirmwareVersion = (TextView) findViewById(R.id.firmware_version);
		txtHardwareVersion = (TextView) findViewById(R.id.hardware_version);
		txtAddress = (TextView) findViewById(R.id.address);
		txtGlobalBand = (TextView) findViewById(R.id.global_band);
		txtSerialNumber = (TextView) findViewById(R.id.serial_number);
		btnInventory = (Button) findViewById(R.id.inventory);
		btnInventory.setOnClickListener(this);
		btnInventory.setOnTouchListener(this);
		btnReadMemory = (Button) findViewById(R.id.read_memory);
		btnReadMemory.setOnClickListener(this);
		btnReadMemory.setOnTouchListener(this);
		btnWriteMemory = (Button) findViewById(R.id.write_memory);
		btnWriteMemory.setOnClickListener(this);
		btnWriteMemory.setOnTouchListener(this);
		btnTagAccess = (Button) findViewById(R.id.tag_access);
		btnTagAccess.setOnClickListener(this);
		btnTagAccess.setOnTouchListener(this);
		btnOption = (Button) findViewById(R.id.option);
		btnOption.setOnClickListener(this);
		btnOption.setOnTouchListener(this);
		btnHumiture = (Button) findViewById(R.id.humiture);
		btnHumiture.setOnClickListener(this);
		btnHumiture.setOnTouchListener(this);
		btnUpdateFirmware = (Button) findViewById(R.id.update_firmware);
		btnUpdateFirmware.setOnClickListener(this);
		btnUpdateFirmware.setOnTouchListener(this);
		btnRFIDUpdateFirmware = (Button) findViewById(R.id.rfid_update_firmware);
		btnRFIDUpdateFirmware.setOnClickListener(this);
		btnRFIDUpdateFirmware.setOnTouchListener(this);
		btnBarcodeScan = (Button) findViewById(R.id.barcode_scan);
		btnBarcodeScan.setOnClickListener(this);
		btnBarcodeScan.setOnTouchListener(this);
		imgBattery = (ImageView) findViewById(R.id.battery);
		imgLogo = (ImageView) findViewById(R.id.app_logo);
		imgLogo.setOnClickListener(this);

		setButtonTextColor(COLOR_DISABLED);
		
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. initWidgets()");
	}

	// Display Application Verserion
	private void displayVersion() {
		String versionName = "";
		String packageName = getPackageName();

		// Get Application Version
		try {
			versionName = getPackageManager().getPackageInfo(packageName, PackageManager.GET_META_DATA).versionName;
		} catch (NameNotFoundException e) {
			versionName = "";
		}
		txtDemoVersion.setText(versionName);

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. displayVersion() - [%s]", versionName);
	}

	// Setup Always Wakeup
	@SuppressLint("InvalidWakeLockTag")
	@SuppressWarnings("deprecation")
	private void setupAlwaysWakeup() {
		if (wakeLock != null)
			return;

		PowerManager powerManager = (PowerManager) this.getSystemService(Activity.POWER_SERVICE);
		if(powerManager != null) {
			wakeLock = powerManager.newWakeLock(PowerManager.FULL_WAKE_LOCK, APP_NAME);
			wakeLock.acquire(1000 );
		}

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setupAlwayWakeup()");
	}

	// Set Default Wakeup
	private void defaultWakeup() {
		if (wakeLock == null)
			return;

		if(wakeLock.isHeld())
			wakeLock.release();

		wakeLock = null;

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. defaultWakeup()");
	}

	// Initialize RFID Reader
	private void initAsReader() {
		mReader = AsReaderP252BManager.getInstance();
		if(mReader == null) {
			Toast.makeText(this, "getInstance failed", Toast.LENGTH_SHORT).show();
			return;
		}
		
		mReader.setEventListener(this);
		mHandler = new Handler();
		mHandler.post(mConnectDevice);
	}
	
	private Runnable mConnectDevice = new Runnable() {
		@Override
		public void run() {
			if (mAsReaderP252BDevice instanceof AsReaderP252BDeviceBluetoothCdc)
				return;

			mAsReaderP252BDevice = mReader.getCurrentDevice();
			if (asReaderP252BUserConfigs.getConnectType() == AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_USB) {
				SharedPreferences ps = getSharedPreferences("USBDevice", Context.MODE_PRIVATE);
				String deviceAddress = ps.getString("usb", "");
				if (!deviceAddress.equals("")) {
					if (mAsReaderP252BDevice == null) {
						mAsReaderP252BDevice = usbCdc;
						if (mReader != null) {
							mReader.connectDevice(mAsReaderP252BDevice);
						}
						asReaderP252BUserConfigs.setConnectType(AsReaderP252BUserConfigs.AsReaderP252BConnectType.AsReaderP252BConnectType_USB);
					}
				}
			} else {
				onStateChanged(mReader, AsReaderP252BConnectionState.Connected);
			}
		}
	};

	// Enable/Disable All Menu Button
	private void enableMenuButtons(boolean enabled) {
		btnInventory.setEnabled(enabled);
		btnInventory.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnReadMemory.setEnabled(enabled);
		btnReadMemory.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnWriteMemory.setEnabled(enabled);
		btnWriteMemory.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnTagAccess.setEnabled(enabled);
		btnTagAccess.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnOption.setEnabled(enabled);
		btnOption.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnHumiture.setEnabled(enabled);
		btnHumiture.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnUpdateFirmware.setEnabled(enabled);
		btnUpdateFirmware.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnRFIDUpdateFirmware.setEnabled(enabled);
		btnRFIDUpdateFirmware.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		btnBarcodeScan.setEnabled(enabled);
		btnBarcodeScan.setTextColor(enabled ? COLOR_ENABLED : COLOR_DISABLED);
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. enableMenuButtons(%s)", enabled);
	}

	// Visible Menu Item
	private void visibleMenuItem(AsReaderP252BConnectionState state) {
		if (state == AsReaderP252BConnectionState.Connected) {
			if (mnuConnectUsbDevice != null)
				mnuConnectUsbDevice.setVisible(false);
			if (mnuConnectBluetoothDevice != null)
				mnuConnectBluetoothDevice.setVisible(false);
			if (mnuBluetoothDisconnect != null && deviceBluetoothCdc.getDeviceBluetooth().connectedDevice() != null)
				mnuBluetoothDisconnect.setVisible(true);
			else if (mnuDisconnectDevice != null)
				mnuDisconnectDevice.setVisible(true);
		} else {
			if (mnuConnectUsbDevice != null)
				mnuConnectUsbDevice.setVisible(true);
			if (mnuConnectBluetoothDevice != null)
				mnuConnectBluetoothDevice.setVisible(true);
			if (mnuBluetoothDisconnect != null)
				mnuBluetoothDisconnect.setVisible(false);
			if (mnuDisconnectDevice != null)
				mnuDisconnectDevice.setVisible(false);
		}

	}

	// Load Configuration
	private void loadConfig() {
		SharedPreferences prefs = getSharedPreferences(APP_NAME, MODE_PRIVATE);
		mBatteryInterval = prefs.getInt(KEY_BATTERY_INTERVAL, DEFAULT_BATTERY_INTERVAL);
		if (mBatteryInterval > 60)
			mBatteryInterval = DEFAULT_BATTERY_INTERVAL;
		AsReaderP252BLog.d(TAG, AsReaderP252BLog.INFO, "DEBUG. loadConfig() - Batery Check Interval : [%d]", mBatteryInterval);
	}

	// Save Configuration
	private void saveConfig() {
		SharedPreferences prefs = getSharedPreferences(APP_NAME, MODE_PRIVATE);
		SharedPreferences.Editor editor = prefs.edit();

		editor.putInt(KEY_BATTERY_INTERVAL, mBatteryInterval);
		AsReaderP252BLog.d(TAG, AsReaderP252BLog.INFO, "DEBUG. saveConfig() - Batery Check Interval : [%d]", mBatteryInterval);
		editor.apply();
	}

	private void startCheckBattery() {
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. startCheckBattery()");
		mBatteryInterval = mUserConfigs.getBatteryInterval();
		int interval = mBatteryInterval * 1000;
		if (interval == 0)
		{
			checkBattery();
			return;
		}
		if(timerCheckBattery != null)
		{
			timerCheckBattery.cancel();
			timerCheckBattery = null;
		}
		timerCheckBattery = new Timer();
		timerCheckBattery.schedule(new TimerTask() {

			@Override
			public void run() {
				runOnUiThread(mActionCheckBattery);
			}

		}, 1000, interval);
	}

	private void stopCheckBattery() {
		if (timerCheckBattery == null)
			return;
		timerCheckBattery.cancel();
		timerCheckBattery = null;
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. stopCheckBattery()");
	}

	private Runnable mActionCheckBattery = new Runnable() {

		@Override
		public void run() {
			if (timerCheckBattery != null)
				checkBattery();
		}

	};

	private void checkBattery() {

		int battery;

		try {
			battery = mReader.getBatteryStatus();
		} catch (AsReaderP252BException e) {
			battery = 0;
		}

		switch (battery) {
		case 0:
			imgBattery.setImageResource(R.drawable.battery0);
			break;
		case 1:
			imgBattery.setImageResource(R.drawable.battery1);
			break;
		case 2:
			imgBattery.setImageResource(R.drawable.battery2);
			break;
		case 3:
			imgBattery.setImageResource(R.drawable.battery3);
			break;
		case 4:
			imgBattery.setImageResource(R.drawable.battery4);
			break;
		default:
			return;
		}
		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. checkBattery() - [%d]", battery);
	}

	private void setButtonTextColor(int color) {
		btnInventory.setTextColor(color);
		btnReadMemory.setTextColor(color);
		btnWriteMemory.setTextColor(color);
		btnTagAccess.setTextColor(color);
		btnOption.setTextColor(color);
		btnHumiture.setTextColor(color);
		btnUpdateFirmware.setTextColor(color);
		btnRFIDUpdateFirmware.setTextColor(color);
		btnBarcodeScan.setTextColor(color);
	}
	
	private void setDisconnectedState() {

		stopCheckBattery();

		txtFirmwareVersion.setText("");
		txtHardwareVersion.setText("");
		txtAddress.setText("");
		txtSerialNumber.setText("");
		txtGlobalBand.setText("");

		imgBattery.setImageResource(R.drawable.battery0);
		enableMenuButtons(false);
		imgLogo.setImageResource(R.drawable.asreader_log_off);
		visibleMenuItem(AsReaderP252BConnectionState.Disconnected);
		WaitDialog.hide();

		AsReaderP252BLog.i(TAG, AsReaderP252BLog.INFO, "INFO. setDisconnectedState()");
	}

	private String getRequestCode(int requestCode) {
		switch (requestCode) {
		case VIEW_INVENTORY:
			return "VIEW_INVENTORY";
		case AccessActivity.VIEW_READ_MEMORY:
			return "VIEW_READ_MEMORY";
		case AccessActivity.VIEW_WRITE_MEMORY:
			return "VIEW_WRITE_MEMORY";
		case AccessActivity.VIEW_TAG_ACCESS:
			return "VIEW_TAG_ACCESS";
		case VIEW_OPTION:
			return "VIEW_OPTION";
		case VIEW_BARCODE_DEMO:
			return "VIEW_BARCODE_DEMO";
		case VIEW_BARCODE_OPTION:
			return "VIEW_BARCODE_OPTION";
		}
		return "";
	}

	private String getResultCode(int resultCode) {
		switch (resultCode) {
		case Activity.RESULT_CANCELED:
			return "RESULT_CANCELED";
		case Activity.RESULT_OK:
			return "RESULT_OK";
		case Activity.RESULT_FIRST_USER:
			return "RESULT_FIRST_USER";
		case RESULT_DISCONNECTED:
			return "RESULT_DISCONNECTED";
		}
		return "";
	}

	private String getMenuId(int id) {
		switch (id) {
			case R.id.menu_search_bluetooth:
				return "menu_search_device";
		}

		return "";
	}

	private String getOnClickId(int id) {
		switch (id) {
		case R.id.inventory:
			return "inventory";
		case R.id.read_memory:
			return "read_memory";
		case R.id.write_memory:
			return "write_memory";
		case R.id.tag_access:
			return "tag_access";
		case R.id.option:
			return "option";
		case R.id.update_firmware:
			return "update_firmware";
		}
		return "";
	}

	@SuppressLint({ "NewApi", "ClickableViewAccessibility" })
	@Override
	public boolean onTouch(View v, MotionEvent event) {
	
		int id = v.getId();
		int action  = event.getAction();
		switch(id) {
		case R.id.inventory:
			switch(action) {
			case MotionEvent.ACTION_DOWN:
				btnInventory.setBackgroundColor(Color.LTGRAY);
				break;
			case MotionEvent.ACTION_UP:
			case MotionEvent.ACTION_CANCEL:
				btnInventory.setBackgroundColor(Color.TRANSPARENT);
				break;
			}
			break;
		case R.id.read_memory:
			switch(action) {
			case MotionEvent.ACTION_DOWN:
				btnReadMemory.setBackgroundColor(Color.LTGRAY);
				break;
			case MotionEvent.ACTION_UP:
			case MotionEvent.ACTION_CANCEL:
				btnReadMemory.setBackgroundColor(Color.TRANSPARENT);
				break;
			}
			break;
		case R.id.write_memory:
			switch(action) {
			case MotionEvent.ACTION_DOWN:
				btnWriteMemory.setBackgroundColor(Color.LTGRAY);
				break;
			case MotionEvent.ACTION_UP:
			case MotionEvent.ACTION_CANCEL:
				btnWriteMemory.setBackgroundColor(Color.TRANSPARENT);
				break;
			}
			break;
			case R.id.tag_access:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnTagAccess.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnTagAccess.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
			case R.id.option:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnOption.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnOption.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
			case R.id.humiture:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnHumiture.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnHumiture.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
			case R.id.firmware_version:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnUpdateFirmware.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnUpdateFirmware.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
			case R.id.rfid_update_firmware:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnRFIDUpdateFirmware.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnRFIDUpdateFirmware.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
			case R.id.barcode_scan:
				switch (action) {
					case MotionEvent.ACTION_DOWN:
						btnBarcodeScan.setBackgroundColor(Color.LTGRAY);
						break;
					case MotionEvent.ACTION_UP:
					case MotionEvent.ACTION_CANCEL:
						btnBarcodeScan.setBackgroundColor(Color.TRANSPARENT);
						break;
				}
				break;
		}
		
		return false;
	}

	public class InitRunnable implements Runnable {

		private final String _firmwareVersion;
		private final String _hardwareVersion;
		private final String _serialNumber;
		private final String _rfVersion;
		private final String _globalBand;

		public InitRunnable(String firmwareVersion,String hardwareVersion,String serialNumber, String rfVersion, String globalBand) {
			_firmwareVersion = firmwareVersion;
			_hardwareVersion = hardwareVersion;
			_serialNumber = serialNumber;
			_rfVersion = rfVersion;
			_globalBand = globalBand;
		}

		@Override
		public void run() {
			txtFirmwareVersion.setText(_firmwareVersion);
			txtHardwareVersion.setText(_hardwareVersion);
			txtSerialNumber.setText(_serialNumber);
			txtAddress.setText(_rfVersion);
			txtGlobalBand.setText(_globalBand);

			startCheckBattery();
			WaitDialog.hide();
			visibleMenuItem(AsReaderP252BConnectionState.Connected);
			imgLogo.setImageResource(R.drawable.asreader_log_on);
			enableMenuButtons(true);
			setProgressBarIndeterminateVisibility(false);
		}
	}

	@Override
	public boolean onKeyDown(int keyCode, KeyEvent event) {
		if (keyCode == KeyEvent.KEYCODE_BACK) {
		    moveTaskToBack(true);
			return false;
		}
		return false;
	}
}
