package com.asreader.jvmasample;

import static jp.co.asterisk.asreader.a60d.sdk.device.DeviceSettingModel.DEV_SETTING_BEEP;
import static jp.co.asterisk.asreader.a60d.sdk.device.DeviceSettingModel.DEV_SETTING_LED;
import static jp.co.asterisk.asreader.a60d.sdk.device.DeviceSettingModel.DEV_SETTING_OFF;
import static jp.co.asterisk.asreader.a60d.sdk.device.DeviceSettingModel.DEV_SETTING_VIB;
import static jp.co.asterisk.asreader.a60d.sdk.device.DeviceSettingModel.DEV_SETTING_ON;

import androidx.appcompat.app.AppCompatActivity;

import android.app.AlertDialog;
import android.bluetooth.BluetoothDevice;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Color;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.provider.Settings;
import android.view.View;
import android.widget.Button;
import android.widget.ProgressBar;
import android.widget.ScrollView;
import android.widget.Switch;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.co.asterisk.asreader.a60d.sdk.AsReaderSDK;
import jp.co.asterisk.asreader.a60d.sdk.device.connection.ConnectionType;
import jp.co.asterisk.asreader.a60d.sdk.device.connection.PermissionState;
import jp.co.asterisk.asreader.a60d.sdk.event.DeviceManagerCallback;
import jp.co.asterisk.asreader.a60d.sdk.event.ASKManagerCallBack;
import jp.co.asterisk.asreader.a60d.sdk.event.BLEScanCallback;
import jp.co.asterisk.asreader.a60d.sdk.protocol.JVMAModel;
import jp.co.asterisk.asreader.a60d.sdk.protocol.JVMARecord;
import jp.co.asterisk.asreader.a60d.sdk.type.PacketValue;
import jp.co.asterisk.asreader.a60d.sdk.type.ask.TriggerStatus;
import jp.co.asterisk.asreader.a60d.sdk.type.jvma.JVMADataId;
import jp.co.asterisk.asreader.a60d.sdk.util.SDKLog;


public class MainActivity extends AppCompatActivity implements DeviceManagerCallback, BLEScanCallback, ASKManagerCallBack{

    private TextView logTextView;
    private TextView statusTextView;
    private TextView batteryTextView;
    private Switch beepSwitch;
    private Switch vibSwitch;
    private Switch ledSwitch;
    private TextView setTextView;
    private TextView clearTextView;
    private TextView gvdTextView;
    private TextView svmdTextView;
    private TextView sicTextView;
    private TextView cvdTextView;
    private TextView cvdvcTextView;
    private TextView acTextView;
    private TextView modeUsbButton;
    private TextView modeBleButton;
    private TextView bleScanButton;
    private ScrollView scrollView;
    private boolean isScrollBottom = true;
    private AlertDialog alertDialog;
    private BleListViewDialog bleListDialog;
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        logTextView = findViewById(R.id.log_text_view);
        statusTextView = findViewById(R.id.status_text_view);
        batteryTextView = findViewById(R.id.battery_text_view);
        beepSwitch = findViewById(R.id.beep_switch);
        vibSwitch = findViewById(R.id.vib_switch);
        ledSwitch = findViewById(R.id.led_switch);
        setTextView = findViewById(R.id.set_text_view);
        clearTextView = findViewById(R.id.clear_text_view);
        gvdTextView = findViewById(R.id.gvd_text_view);
        svmdTextView = findViewById(R.id.svmd_text_view);
        sicTextView = findViewById(R.id.sic_text_view);
        cvdTextView = findViewById(R.id.cvd_text_view);
        cvdvcTextView = findViewById(R.id.cvdvc_text_view);
        acTextView = findViewById(R.id.ac_text_view);
        modeUsbButton = findViewById(R.id.mode_usb_button);
        modeBleButton = findViewById(R.id.mode_ble_button);
        bleScanButton = findViewById(R.id.ble_scan_button);
        scrollView = findViewById(R.id.scroll_view);
        initAction();

        //以下のデリゲートを実現する必要があります
        AsReaderSDK.getInstance().getASKManager().setAskManagerCallBack(this);
        AsReaderSDK.getInstance().getDeviceManager().setDeviceManagerCallback(this);

        setConnectionModule(ConnectionType.USB);

        //ログ機能のオン
        AsReaderSDK.getInstance().changeSDKLogType(SDKLog.LogType.LogType_APP);
        AsReaderSDK.getInstance().setLogLevel(SDKLog.LogLevel.Debug);
        AsReaderSDK.getInstance().setLog(true);
    }

    /**
     * UIPart
     */
    public void initAction(){
        setTextView.setOnClickListener ( v -> {
            setSettingParametar();
        });

        clearTextView.setOnClickListener( v -> {
            logTextView.setText("");
        });
        gvdTextView.setOnClickListener( v -> {
            getVmDataSample();
        });
        svmdTextView.setOnClickListener( v -> {
            setVmManageDataSample();
        });
        sicTextView.setOnClickListener( v -> {
            setItemCodeSample();
        });
        cvdTextView.setOnClickListener( v -> {
            clearVmDataSample();
        });
        cvdvcTextView.setOnClickListener( v -> {
            clearVmDataWithoutConfirmationSample();
        });
        acTextView.setOnClickListener( v -> {
            allClearSample();
        });

        modeUsbButton.setOnClickListener( v -> {
            AsReaderSDK.getInstance().getDeviceManager().destroy();
            setConnectionModule(ConnectionType.USB);
        });

        modeBleButton.setOnClickListener( v -> {
            AsReaderSDK.getInstance().getDeviceManager().destroy();
            setConnectionModule(ConnectionType.BLE);
        });

        bleScanButton.setOnClickListener( v -> {
            showScanPopUp();
        });
        beepSwitch.setOnCheckedChangeListener( (buttonView, isChecked) -> {

        });
        vibSwitch.setOnCheckedChangeListener( (buttonView, isChecked) -> {

        });
        ledSwitch.setOnCheckedChangeListener( (buttonView, isChecked) -> {

        });
        scrollView.setOnScrollChangeListener( (View v, int scrollX, int scrollY, int oldScrollX, int oldScrollY) -> {
            View content = scrollView.getChildAt(0);
            if (content == null) return;

            isScrollBottom = (scrollY + scrollView.getHeight()) >= content.getHeight();
        });
    }

    private void addToLogText(String logText) {
        logTextView.setText(logTextView.getText().toString()+logText);
        if (isScrollBottom) {
            scrollView.post(() -> {
                scrollView.fullScroll(View.FOCUS_DOWN);
            });
        }
    }

    private void setConnectionModule(ConnectionType connectionType){
        if(connectionType == ConnectionType.BLE){
            bleScanButton.setVisibility(View.VISIBLE);
            modeBleButton.setBackgroundColor(Color.CYAN);
            modeBleButton.setTextColor(Color.BLACK);
            modeUsbButton.setBackgroundColor(Color.WHITE);
            modeUsbButton.setTextColor(Color.GRAY);
        }

        if(connectionType == ConnectionType.USB){
            bleScanButton.setVisibility(View.GONE);
            modeUsbButton.setBackgroundColor(Color.CYAN);
            modeUsbButton.setTextColor(Color.BLACK);
            modeBleButton.setBackgroundColor(Color.WHITE);
            modeBleButton.setTextColor(Color.GRAY);
        }

        initializeModule(connectionType);
    }

    /**
     * BLE Scan UI
     */
    public void showScanPopUp() {
        //すべてAsReaderとの接続を切断
        AsReaderSDK.getInstance().closeBle();

        if (bleListDialog == null) {
            bleListDialog = new BleListViewDialog(this, R.layout.listview_ble_dialog);
        }
        final ProgressBar progressBar = this.bleListDialog.findViewById(R.id.progresscircle);
        bleListDialog.clearData();
        progressBar.setVisibility(View.INVISIBLE);
        bleListDialog.setOnDismissListener( d -> {
            //BLEデバイスの検索停止
            AsReaderSDK.getInstance().stopBLEScanDevice();
        });
        bleListDialog.onOnSetItemClickListener( i -> {
            BleListViewDialog.ListData dataIndex = bleListDialog.getDataIndex(i);
            if (dataIndex != null) {
                //BLEデバイスの検索停止
                AsReaderSDK.getInstance().stopBLEScanDevice();
                //BLEデバイスに接続
                AsReaderSDK.getInstance().open(dataIndex.getBluetoothDevice());
            }
            bleListDialog.dismiss();
        });
        bleListDialog.findViewById(R.id.btn_cancel).setOnClickListener( v -> {
            bleListDialog.dismiss();
        });
        bleListDialog.findViewById(R.id.btn_search).setOnClickListener( v -> {
            progressBar.setVisibility(View.VISIBLE);
            bleListDialog.clearData();
            //BLEデバイスの検索開始
            AsReaderSDK.getInstance().startBLEScanDevice();
            new Handler().postDelayed(new Runnable() {
                public void run() {
                    progressBar.setVisibility(View.INVISIBLE);
                }
            }, 3000); // scan for 3 sec and stops
        });
        this.bleListDialog.show();
    }

    /**
     * BLE権限
     *
     * いずれか一つの許可が永久拒否された場合、ダイアロッグを表示します。
     * @param permissionState 権限のステータス（値は永久拒否の一つ）.
     */
    private void showPermissionDialog(PermissionState permissionState) {
        if(alertDialog !=null) alertDialog.dismiss();

        String title = "";
        String message = "Permission has been denied more than once. \nPlease allow it from app setting page";

        switch (permissionState){

            case LOCATION_PERMISSION_DENIED_FOREVER:
                title = "Location Permission";
                break;
            case BLUETOOTH_PERMISSION_DENIED_FOREVER:
                title = "Bluetooth Permission";
                break;
        }

        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle(title).setMessage(message);
        builder.setPositiveButton("OK", (d,i) -> {
            /**
             * Opens application settings in Android Settings app.
             */
            final Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS);
            intent.setData(Uri.fromParts("package", getPackageName(), null));
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            startActivity(intent);
        });
        alertDialog = builder.create();
        alertDialog.show();
    }

    /**
     * BLE/USBで接続の設定
     */
    private void initializeModule(ConnectionType connectionType){
        if(connectionType == ConnectionType.BLE){
            // To use BLE Step 1 : declare connection type to BLE
            AsReaderSDK.getInstance().initialize(getApplicationContext(), ConnectionType.BLE);

            // To use BLE Step 2 : Set BLE scan callback
            AsReaderSDK.getInstance().setBLEScanCallback(this);

            // To use BLE Step 3 : Check BLE permission from the app
            PermissionState permissionState = AsReaderSDK.getInstance().checkBlePermission(this);

            // To use BLE Step 4 : If one of the BLE permission is denied forever, show dialog to manually allow permission
            if(permissionState == PermissionState.LOCATION_PERMISSION_DENIED_FOREVER ||
                    permissionState == PermissionState.BLUETOOTH_PERMISSION_DENIED_FOREVER) {
                showPermissionDialog(permissionState);
            }
        }

        if(connectionType == ConnectionType.USB){
            // To use USB Step : declare connection type to USB
            AsReaderSDK.getInstance().initialize(getApplicationContext(), ConnectionType.USB);
        }
    }

    /**
     * 音声、振動、LEDのOn/Offの設定
     */
    public void setSettingParametar(){
        String beepSetting = beepSwitch.isChecked()?DEV_SETTING_ON:DEV_SETTING_OFF;
        AsReaderSDK.getInstance().getDeviceManager().setSettingParametar(DEV_SETTING_BEEP,beepSetting);
        String vibSetting = vibSwitch.isChecked()?DEV_SETTING_ON:DEV_SETTING_OFF;
        AsReaderSDK.getInstance().getDeviceManager().setSettingParametar(DEV_SETTING_VIB,vibSetting);
        String lebSetting = ledSwitch.isChecked()?DEV_SETTING_ON:DEV_SETTING_OFF;
        AsReaderSDK.getInstance().getDeviceManager().setSettingParametar(DEV_SETTING_LED,lebSetting);
        //デバイスの設定を確定します。
        AsReaderSDK.getInstance().getDeviceManager().commit();
    }

    /**
     * ASKManager コールバック
     */

    /**
     * データ受信時にコールバックされます。
     * @param result 受信データ
     * @param ackNakResult ACK/NAKの種類
     */
    @Override
    public void onReceivedJVMAResult(List<JVMARecord> result, JVMAModel ackNakResult) {
        runOnUiThread(() -> {
            if (result != null) {
                for (JVMARecord jvmaRecord : result) {
                    addToLogText("Record identifyingCode: " + jvmaRecord.getIdentifyingCode() + "\r\n");
                    addToLogText("Record recordLength: " + jvmaRecord.getRecordLength() + "\r\n");
                    addToLogText("Record itemDigitNumber: " + jvmaRecord.getItemDigitNumber() + "\r\n");

                    for (String item : jvmaRecord.getItems()) {
                        addToLogText("Record item: " + item + "\r\n");
                    }
                }
            }
            if (ackNakResult.getPacketType() == PacketValue.RESPONSE_ACK) {
                addToLogText("onReceivedJVMAResult ACK: " + ackNakResult.getAckStatus() + "\r\n");
            } else if (ackNakResult.getPacketType() == PacketValue.RESPONSE_NAK) {
                addToLogText("onReceivedJVMAResult NAK: " + ackNakResult.getAckStatus() + "\r\n");
            }
        });
    }

    /**
     * エラー時にコールバックします。
     * @param message エラーメッセージ
     */
    @Override
    public void onASKManagerError(String message) {
        runOnUiThread(() -> addToLogText("onASKManagerError:" + message + "\r\n"));
    }

    /**
     * 送信、受信コマンドのメッセージをコールバックします。
     * @param message コマンドメッセージ
     */
    @Override
    public void onReceiveASKCommandMessage(String message) {
        runOnUiThread(() -> addToLogText("onReceiveASKCommandMessage:" + message + "\r\n"));
    }

    /**
     * BLEScan コールバック
     */
    /**
     * BEL検索／BLE接続する時にエラーが発生する場合のコールバック関数
     * @param errorMsg 詳細のエラーメッセージ
     */
    @Override
    public void onBLEScanError(String errorMsg) {
        runOnUiThread(() -> addToLogText("onBLEScanError:" + errorMsg + "\r\n"));
    }

    /**
     *  {AsReaderSDK.getInstance().startBLEScanDevice()}のコールバックメソッド
     * @param bluetoothDevice bluetoothデバイス検索
     * @param name bluetoothデバイス名の検索
     * @param rssi デバイスrssiの検索
     */
    @Override
    public void onBLEScanResult(BluetoothDevice bluetoothDevice, String name, String rssi) {
        runOnUiThread(() -> bleListDialog.addData(bluetoothDevice, name, rssi));
    }

    /**
     * DeviceManager コールバック
     */
    /**
     * デバイス接続時にコールバックされます。
     * @param isConnect　接続：true　切断：false
     */
    @Override
    public void onConnect(boolean isConnect) {
        runOnUiThread(() -> {
            addToLogText("onConnect:" + isConnect + "\r\n");
            String text = isConnect ? "Connect" : "Disconnect";
            statusTextView.setText(text);
            if (!isConnect){
                onReceivedBattery(0);
            }
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            //電源On/Offをする
            AsReaderSDK.getInstance().getDeviceManager().setPower(isConnect);
        });
    }

    /**
     * パラメータの設定完了時にコールバックされます。
     * @param isPowerOn　電源ON：true　電源OFF：false
     */
    @Override
    public void onCompletedPreparation(boolean isPowerOn) {
        runOnUiThread(() -> addToLogText("onCompletedPreparation:" + isPowerOn + "\r\n"));
    }

    /**
     * バッテリーの電量通知時にコールバックされます。
     * @param battery　バッテリー電量
     */
    @Override
    public void onReceivedBattery(int battery) {
        runOnUiThread(() -> batteryTextView.setText(battery+""));
    }
    /**
     * トリガーボタンイベント時にコールバックされます。
     * @param status　トリガーボタンイベント
     */
    @Override
    public void onReceivedTriggerState(TriggerStatus status) {
        runOnUiThread(() -> addToLogText("onReceivedTriggerState:" + status + "\r\n"));
    }

    /**
     * デバイスエラー時にコールバックされます。
     * @param message　エラーメッセージ
     */
    @Override
    public void onDeviceManagerError(String message) {
        runOnUiThread(() -> addToLogText("onDeviceManagerError:" + message + "\r\n"));
    }

    /**
     * モジュールバージョン情報取得完了時にコールバックされます。
     * @param versionText　モジュールバージョン
     */
    @Override
    public void onReceiveModuleVersion(String versionText) {
        runOnUiThread(() -> addToLogText("onReceiveModuleVersion:" + versionText + "\r\n"));
    }
    /**
     * デバイス設定完了時にコールバックされます。
     * @param params　デバイス設定
     */
    @Override
    public void onReceivedSettingParameter(Map<String, String> params) {
        runOnUiThread(() -> {
            addToLogText("onReceivedSettingParameter:\n");
            for (Map.Entry<String,String> entry : params.entrySet()) {
                addToLogText(entry.getKey() +":" + entry.getValue() + "\n");
            }
        });
    }

    /**
     *  SDK-APISample
     */
    public void getVmDataSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //収集したいDataIDを作成します（複数作成可能）。
        List<String> dataRecords = new ArrayList<>();
        dataRecords.add(JVMADataId.InputCashQuantity.getName());
        dataRecords.add(JVMADataId.QuitCashQuantity.getName());
        ////SDKメソッドデータ収集を呼び出します。
        AsReaderSDK.getInstance().getASKManager().getVmData(true,dataRecords);
    }
    public void setVmManageDataSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //dataを作成します（複数作成可能）。
        List<String> vmManageDataList = new ArrayList<>();
        vmManageDataList.add("1000000000");
        vmManageDataList.add("2000000000");

        //設定データを作成します（複数作成可能）
        Map<JVMADataId, List<String>> settingData = new HashMap<>();
        //自販機管理データレコード"
        settingData.put(JVMADataId.ManageData, vmManageDataList);
        //自販機に設定データを設定します。
        AsReaderSDK.getInstance().getASKManager().setSettingDataList(settingData);
        AsReaderSDK.getInstance().getASKManager().sendSettingData(true);
    }
    public void setItemCodeSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //itemを作成します（複数作成可能）。
        List<String> itemCodeList = new ArrayList<>();
        itemCodeList.add("10513800000000");
        itemCodeList.add("12004400000000");
        itemCodeList.add("10500400000000");

        //設定データを作成します（複数作成可能）
        Map<JVMADataId, List<String>> settingData = new HashMap<>();
        //商品コードレコード
        settingData.put(JVMADataId.ShouhinCD, itemCodeList);
        //自販機に設定データを設定します。
        AsReaderSDK.getInstance().getASKManager().setSettingDataList(settingData);
        AsReaderSDK.getInstance().getASKManager().sendSettingData(true);
    }
    public void clearVmDataSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //収集したいDataIDを作成します。
        List<String> dataRecords = new ArrayList<>();
        dataRecords.add(JVMADataId.InputCashQuantity.getName());
        dataRecords.add(JVMADataId.QuitCashQuantity.getName());
        // 自販機データを収集しクリアして、その後確認します。
        // clearVMData実行時には02通信完了後SDKからconfirmClearが呼び出される。クリア確認結果を返します。
        AsReaderSDK.getInstance().getASKManager().clearVmData(true,dataRecords);
    }

    public void clearVmDataWithoutConfirmationSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //収集したいDataIDを作成します。
        List<String> dataRecords = new ArrayList<>();
        dataRecords.add(JVMADataId.InputCashQuantity.getName());
        dataRecords.add(JVMADataId.QuitCashQuantity.getName());
        //自販機データを収集し、その後自販機データをクリアします。クリア結果を返します。
        AsReaderSDK.getInstance().getASKManager().clearVmDataWithoutConfirmation(true,dataRecords);
        //クリア結果が出てから、必要に応じて、自販機データ収集クリア確認を行います。（02クリア確認）
        //AsReaderSDK.getInstance().getASKManager().confirmClear();
    }
    public void allClearSample(){
        AsReaderSDK.getInstance().getASKManager().setTerminalPassword("1234567890");
        AsReaderSDK.getInstance().getASKManager().setTerminalSettingCode("1234567890");
        //自販機のデータをすべてクリアします。通信完了後SDKからConfirmAllClearが呼び出される。クリア確認結果を返します。
        AsReaderSDK.getInstance().getASKManager().allClear(true);
    }
}