import 'dart:async';
import 'package:a3xd_plugin/a3xd_plugin.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:a3xd_plugin_example/more.dart';
import 'package:a3xd_plugin_example/dataitem.dart';
import 'package:a3xd_plugin_example/LocalCache.dart';
import 'event_bus_instance.dart';

void main() => runApp(const MaterialApp(
      title: 'AsReader',
      home: MyApp(),
      debugShowCheckedModeBanner: false,
    ));

class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    var home = const Scaffold(
      body: MainWidget(),
    );
    return home;
  }
}

class MainWidget extends StatefulWidget {
  const MainWidget({super.key});
  @override
  State<StatefulWidget> createState() => MainState();
}

class MainState extends State<MainWidget> {
  final _a3xdPlugin = A3xdPlugin();
  static const iosCallPlatform = MethodChannel('a3xd_plugin');
  final cache = LocalCache();
  bool isBeep = false;
  bool isVibration = false;
  bool isPower = false;
  bool isEnabled = false;
  bool isSetting = false;
  bool isScan = false;
  Color startBackgroundColor = Colors.grey;
  Color startTitleColor = Colors.black54;
  Color stopBackgroundColor = Colors.grey;
  Color stopTitleColor = Colors.black54;
  Color clearBackgroundColor = Colors.grey;
  Color clearTitleColor = Colors.black54;
  List<DataItem> listItems = [];
  Map<String, DataItem> map = {};
  int totalCount = 0;
  String connectStatus = 'Disconnected';
  String batteryStatus = '0%';
  Widget moreIcon = const Icon(null);
  late BuildContext mContext;
  late StreamSubscription _subscription;

  @override
  void initState() {
    super.initState();
    initPlatformState();
    _subscription = eventBus.on<DisconnectEvent>().listen((event) {
      connected(false);
    });
  }

  @override
  void dispose() {
    _subscription.cancel();
    super.dispose();
  }

  Future<void> initPlatformState() async {
    Future<dynamic> _handler(MethodCall methodCall) {
      print('main methodCall.method:${methodCall.method}, methodCall.arguments:${methodCall.arguments}');
      if (_a3xdPlugin.ON_CONNECT == methodCall.method) {
        bool isConnected = methodCall.arguments['result'];
        print('onConnect:$isConnected');
        connected(isConnected);
      } else if (_a3xdPlugin.ON_BATTERY_STATE_RECEIVED == methodCall.method) {
        int battery = methodCall.arguments['dest'];
        print('onBatteryStateReceived:$battery');
        batteryReceived(battery);
      } else if (_a3xdPlugin.ON_TAG_WITH_RSSI_RECEIVED == methodCall.method) {
        List<int> intList = methodCall.arguments['pcEpc'].whereType<int>().toList();
        String hexString = convertToHex(intList);
        int rssi = methodCall.arguments['rssi'];
        print('onTagWithRssiReceived:$hexString  rssi:$rssi');
        receivedData(hexString, rssi.toDouble());
      } else if (_a3xdPlugin.ON_COMPLETED_PREPARATION == methodCall.method) {
        bool isPowerOn = methodCall.arguments['result'];
        print('onCompletedPreparation:$isPowerOn');
        onCompletedPreparation(isPowerOn);
      } else if (_a3xdPlugin.ON_DEVICE_MANAGER_ERROR == methodCall.method) {
        String error = methodCall.arguments['result'];
        print('onDeviceManagerError:$error');
        onDeviceManagerError(error);
       } else if (_a3xdPlugin.ON_RFID_MANAGER_ERROR == methodCall.method) {
         String error = methodCall.arguments['result'];
         print('onRFIDManagerError:$error');
         onDeviceManagerError(error);
       } else if (_a3xdPlugin.ON_AUTO_INVENTORY_FINISHED == methodCall.method) {
        print('ON_AUTO_INVENTORY_FINISHED');
        isScan = false;
        stopBackgroundColor = Colors.grey;
        stopTitleColor = Colors.black54;
        startBackgroundColor = Colors.white;
        startTitleColor = Colors.black;
        setState(() {});
      }
      return Future.value('success');
    }
    iosCallPlatform.setMethodCallHandler(_handler);
  }

  @override
  Widget build(BuildContext context) {
    mContext = context;
    ListView listView;
    var boldGrayContainer = Container(
      color: Colors.black54,
      padding: const EdgeInsets.fromLTRB(15, 7, 15, 7),
      height: 2,
    );
    var upContainer = Container(
      color: Colors.white,
      padding: const EdgeInsets.fromLTRB(30, 7, 30, 7),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceEvenly,
        children: <Widget>[
          Expanded(
            child: Text(
              '$totalCount tags',
              style: const TextStyle(
                  height: 1.5,
                  color: Colors.black,
                  fontSize: 16.0,
                  fontWeight: FontWeight.w400,
                  backgroundColor: Colors.white38),
              textAlign: TextAlign.left,
            ),
          ),
          Expanded(
            child: Text(
              connectStatus,
              style: const TextStyle(
                  height: 1.5,
                  color: Colors.black,
                  fontSize: 16.0,
                  fontWeight: FontWeight.w400,
                  backgroundColor: Colors.white38),
              textAlign: TextAlign.center,
            ),
          ),
          Expanded(
            child: Text(
              batteryStatus,
              style: const TextStyle(
                  height: 1.5,
                  color: Colors.black,
                  fontSize: 16.0,
                  fontWeight: FontWeight.w400,
                  backgroundColor: Colors.white38),
              textAlign: TextAlign.right,
            ),
          ),
        ],
      ),
    );
    var boldGrayContainer2 = Container(
      color: Colors.black54,
      padding: const EdgeInsets.fromLTRB(15, 7, 15, 7),
      height: 2,
    );
    listView = ListView.builder(
        shrinkWrap: true,
        itemCount: listItems.length,
        itemBuilder: (context, index) {
          final listItem = listItems[index];
          return Column(
            children: <Widget>[
              getListTile(listItem.barcode, listItem.rssi, listItem.count),
              const Divider(height: 1, color: Colors.black38),
            ],
          );
        });
    var boldGrayContainer3 = Container(
      color: Colors.black54,
      padding: const EdgeInsets.fromLTRB(15, 7, 15, 7),
      height: 2,
    );
    var bottomContainer = Container(
      color: Colors.white,
      child: Visibility(
        child: Row(
          mainAxisAlignment: MainAxisAlignment.spaceEvenly,
          children: <Widget>[
            Switch(
              value: isPower,
              onChanged: isEnabled
                  ? (newValue) {
                getConnectStatus(newValue);
              }
                  : null,
            ),
            ElevatedButton(
              style: ElevatedButton.styleFrom(
                backgroundColor: startBackgroundColor,
                foregroundColor: startTitleColor,
              ),
              onPressed: _startScan,
              child: const Text('Read'),
            ),
            ElevatedButton(
              style: ElevatedButton.styleFrom(
               backgroundColor: stopBackgroundColor,
               foregroundColor: stopTitleColor,
              ),
              onPressed: _stopScan,
              child: const Text('Stop'),
            ),
            ElevatedButton(
              style: ElevatedButton.styleFrom(
               backgroundColor: clearBackgroundColor,
               foregroundColor: clearTitleColor,
              ),
              onPressed: _clear,
              child: const Text('Clear'),
            ),
          ],
        ),
      ),
    );
    return Scaffold(
      appBar: AppBar(
        title: Text('RFID Demo'),
        backgroundColor: Colors.white,
        actions: <Widget>[
          IconButton(
            icon: moreIcon,
            onPressed: isSetting
                ? () async {
              await Navigator.push(context,
                  MaterialPageRoute(builder: (context) => MoreWidget()));
              initPlatformState();
            }
                : null,
          ),
        ],
      ),
      body: Container(
        color: Colors.white,
        child: Column(
          children: <Widget>[
              boldGrayContainer,
              upContainer,
              boldGrayContainer2,
              Expanded(
                child: listView,
              ),
              boldGrayContainer3,
            const SizedBox(height: 5),
            bottomContainer,
            const SizedBox(height: 5),
          ],
          ),
        ),
      );
  }

  Widget getListTile(String barcode, double rssi, int count) {
    return Container(
      padding: const EdgeInsets.all(5.0),
      child: ListTile(
        onTap: () {},
        title: Text(
          barcode,
          style: const TextStyle(color: Colors.black),
        ),
        subtitle: Text(
          rssi.toString(),
          style: const TextStyle(color: Colors.black54),
        ),
        trailing: Text(
          count.toString(),
          style: const TextStyle(color: Colors.black54),
        ),
      ),
    );
  }

  Future<void> _startScan() async {
    print('_startScan start');
    Future<bool?> isConnect = _a3xdPlugin.isConnected();
    isConnect.then((futureResult) {
      if (futureResult == true) {
        Future<bool?> status = _a3xdPlugin.isM30SDevice();
        status.then((futureResult) {
          if (futureResult == true) {
            if (isScan == false) {
              isScan = true;
              _a3xdPlugin.startInventoryRSSI(0, 0, 0);
              startBackgroundColor = Colors.grey;
              startTitleColor = Colors.black54;
              stopBackgroundColor = Colors.white;
              stopTitleColor = Colors.black;
              setState(() {});
            }
          } else {
            Future<bool?> status = _a3xdPlugin.isPowerOn();
            status.then((futureResult) {
              if (futureResult == true) {
                if (isScan == false) {
                  isScan = true;
                  _a3xdPlugin.startInventoryRSSI(0, 0, 0);
                  startBackgroundColor = Colors.grey;
                  startTitleColor = Colors.black54;
                  stopBackgroundColor = Colors.white;
                  stopTitleColor = Colors.black;
                  setState(() {});
                }
              }
            });
          }
        });
      }
    });
    print('_startScan end');
  }

  Future<void> _stopScan() async {
    print('_stopScan start');
    Future<bool?> isConnect = _a3xdPlugin.isConnected();
        isConnect.then((futureResult) {
          if (futureResult == true) {
            Future<bool?> status = _a3xdPlugin.isM30SDevice();
            status.then((futureResult) {
              if (futureResult == true) {
                if (isScan == true) {
                  isScan = false;
                  _a3xdPlugin.stopDecode();
                  stopBackgroundColor = Colors.grey;
                  stopTitleColor = Colors.black54;
                  startBackgroundColor = Colors.white;
                  startTitleColor = Colors.black;
                  setState(() {});
                }
              } else {
                Future<bool?> status = _a3xdPlugin.isPowerOn();
                status.then((futureResult) {
                  if (futureResult == true) {
                    if (isScan == true) {
                      isScan = false;
                      _a3xdPlugin.stopDecode();
                      stopBackgroundColor = Colors.grey;
                      stopTitleColor = Colors.black54;
                      startBackgroundColor = Colors.white;
                      startTitleColor = Colors.black;
                      setState(() {});
                    }
                  }
                });
              }
            });
          }
    });
    print('_stopScan end');
  }

  void _clear() {
    print('_clear start');
    Future<bool?> isConnect = _a3xdPlugin.isConnected();
        isConnect.then((futureResult) {
        if (futureResult == true) {
          Future<bool?> status = _a3xdPlugin.isM30SDevice();
          status.then((futureResult) {
            if (futureResult == true) {
              listItems.clear();
              map.clear();
              totalCount = 0;
              setState(() {});
            } else {
              Future<bool?> status = _a3xdPlugin.isPowerOn();
              status.then((futureResult) {
                if (futureResult == true) {
                  listItems.clear();
                  map.clear();
                  totalCount = 0;
                  setState(() {});
                }
              });
            }
          });
        }
     });
    print('_clear end');
  }

  void connected(bool isConnected) {
    setState(() {
      if (isConnected) {
        connectStatus = 'Connected';
        getConnectStatus(true);
      } else {
        isScan = false;
        isPower = false;
        isEnabled = false;
        connectStatus = 'Disconnected';
        batteryStatus = '0%';
        moreIcon = const Icon(null);
        isSetting = false;
        startBackgroundColor = Colors.grey;
        startTitleColor = Colors.black54;
        stopBackgroundColor = Colors.grey;
        stopTitleColor = Colors.black54;
        clearBackgroundColor = Colors.grey;
        clearTitleColor = Colors.black54;
      }
    });
  }

  void batteryReceived(int battery) {
    setState(() {
      batteryStatus = '$battery%';
    });
  }

  void receivedData(String decodeData, double rssi) {
    setState(() {
      if (map.containsKey(decodeData)){
        DataItem item = map[decodeData]!;
        item.rssi = rssi;
        item.count = item.count + 1;
        map.update(decodeData, (value) => item);
      } else {
        DataItem item = DataItem(decodeData, rssi, 1);
        map[decodeData] = item;
        listItems.add(item);
      }
      totalCount = totalCount + 1;
    });
  }

  void onCompletedPreparation(bool isPowerOn) {
    setState(() {
       if (isPowerOn) {
          moreIcon = Image.asset(
                'images/settings.png',
                width: 25,
                height: 25,
                fit: BoxFit.cover,
          );
          isSetting = true;
          startBackgroundColor = Colors.white;
          startTitleColor = Colors.black;
          stopBackgroundColor = Colors.white;
          stopTitleColor = Colors.black;
          clearBackgroundColor = Colors.white;
          clearTitleColor = Colors.black;
          setReaderSetting();
       } else {
          isScan = false;
          moreIcon = const Icon(null);
          isSetting = false;
          startBackgroundColor = Colors.grey;
          startTitleColor = Colors.black54;
          stopBackgroundColor = Colors.grey;
          stopTitleColor = Colors.black54;
          clearBackgroundColor = Colors.grey;
          clearTitleColor = Colors.black54;
       }
    });
  }

  void setReaderSetting() async {
    isBeep = await cache.getBeep();
    isVibration = await cache.getVibration();
    _a3xdPlugin.setReaderSettings(isBeep, isVibration, true, true);
    Future<bool?> status = _a3xdPlugin.isM30SDevice();
    status.then((futureResult) {
      if (futureResult == true) {
        setState(() {
          _a3xdPlugin.setContinuousMode(true);
          isPower = false;
          isEnabled = false;
        });
      }
    });
  }

  void onDeviceManagerError(String error) {
    setState(() {
      showAlertDialog(
          mContext, '', error);
    });
  }

  String convertToHex(List<int> intList) {
    List<String> hexParts = [];
    for (int value in intList) {
      String hex = value.toRadixString(16);
      if (hex.length % 2 != 0) {
        hex = '0$hex';
      }
      hex = hex.toUpperCase();
      hexParts.add(hex);
    }
    return hexParts.join('');
  }

  void getConnectStatus(bool power) {
     Future<bool?> status = _a3xdPlugin.isConnected();
        status.then((futureResult) {
          if (futureResult == true) {
            setPower(power);
          }
     });
  }

  void setPower(bool power) async {
     isEnabled = true;
     isPower = power;
     isBeep = await cache.getBeep();
     isVibration = await cache.getVibration();
     _a3xdPlugin.setPowerWithOption(power, isBeep, isVibration, true, true, true);
     setState(() {});
  }

  void showAlertDialog(BuildContext context, String title, String message) {
    Widget okButton = ElevatedButton(
      child: const Text('OK'),
      onPressed: () {
        Navigator.of(context).pop();
      },
    );
    AlertDialog alert = AlertDialog(
      title: Text(title),
      content: Text(message),
      actions: [
        okButton,
      ],
    );
    showDialog(
      context: context,
      builder: (BuildContext context) {
        return alert;
      },
    );
  }
}
