#import "ASR033WProtocol.h"
#import "ASR033W.h"
#import "ASR033WUtility.h"
#import <UIKit/UIKit.h>

// Declare Packet Prefix/Suffix Codes
#define PACKET_PREFIX                   0x7e    // '~'
#define PACKET_SUFFIX1                  0x0d    // '\r'
#define PACKET_SUFFIX2                  0x0a    // '\n'
#define RESULT_CODE_LENGTH              4

@implementation ASR033WProtocol{
    NSMutableData *mRecvData;
    ASR033WPacketManager *mPackets;
    ASR033WCommandType mAction;
    ASR033W *_asr033w;
    ASR033WNetworkTCP *_currentTcp;
    ASR033WNetworkMQTT *_currentMQTT;
}

@synthesize delegate;

- (id)initWithASR033W:(ASR033W *)asreader asr033WNetworkMqtt:(ASR033WNetworkMQTT *)asr033WNetworkMqtt tcp:(ASR033WNetworkTCP *)asP52NNetworkTCP{
    self = [super init];
    if (self) {
        _currentTcp = asP52NNetworkTCP;
        mAction = ASR033WCommandStop;
        _asr033w = asreader;
        mPackets = [ASR033WPacketManager new];
        _isReportRssi = FALSE;
        mRecvData = [NSMutableData new];
        _currentMQTT = asr033WNetworkMqtt;
    }
    return self;
}

- (void)disconnect {
    mAction = ASR033WCommandStop;
    mPackets = [ASR033WPacketManager new];
    _isReportRssi = FALSE;
    mRecvData = [NSMutableData new];
}

- (ASR033WCommandType)getAction {
    return mAction;
}
- (NSData *)getPropertyData:(ASR033WPropertyType)type withParameter:(NSString *)param {
    NSString *data = @"";
    if (param != nil && ![param isEqualToString:@""] && param.length > 0) {
        data = [NSString stringWithFormat:@"~r%c%@\r\n", type, param];
    } else {
        data = [NSString stringWithFormat:@"~r%c\r\n", type];
    }
    NSData *sendData = [data dataUsingEncoding:NSASCIIStringEncoding];
    
    
    ASR033WResultData *res = [self sendDataWithResponse:sendData responsePacketType:ASR033WPacketResGetProperty responseSubType:(uint8_t)type];
    if (res.result != ASR033WResultNoError) {
        return nil;
    }
    return res.data;
}
- (NSString *)getProperty:(ASR033WPropertyType)type withParameter:(NSString *)param {
    NSString *data = @"";
    if (param != nil && ![param isEqualToString:@""] && param.length > 0) {
        data = [NSString stringWithFormat:@"~r%c%@\r\n", type, param];
    } else {
        data = [NSString stringWithFormat:@"~r%c\r\n", type];
    }
    NSData *sendData = [data dataUsingEncoding:NSASCIIStringEncoding];
    
    
    ASR033WResultData *res = [self sendDataWithResponse:sendData responsePacketType:ASR033WPacketResGetProperty responseSubType:(uint8_t)type];
    if (res.result != ASR033WResultNoError) {
        return @"";
    }
    NSString *resData = [[NSString alloc]initWithData:res.data
                                             encoding:NSASCIIStringEncoding];
    return resData;
}
- (ASR033WResultCode)setProperty:(ASR033WPropertyType)type withParameterData:(NSData *)param {
    NSMutableData *sendData = [[NSMutableData alloc] initWithData:[[NSString stringWithFormat:@"~w%c", type] dataUsingEncoding:NSASCIIStringEncoding]];
    if (param && param.length > 0) {
        [sendData appendData:param];
    }
    NSData *endData = [@"\r\n" dataUsingEncoding:NSASCIIStringEncoding];
    [sendData appendData:endData];
    ASR033WResultCode res = [self sendData:sendData responsePacketType:ASR033WPacketResSetProperty responseSubType:(uint8_t)type];
    if (res != ASR033WResultNoError) {
        return res;
    }
    return ASR033WResultNoError;
}

- (ASR033WResultCode)setProperty:(ASR033WPropertyType)type withParameter:(NSString *)param {
    NSString *data = @"";
    if (param != nil && ![param isEqualToString:@""] && param.length > 0) {
        data = [NSString stringWithFormat:@"~w%c%@\r\n", type, param];
    } else {
        data = [NSString stringWithFormat:@"~w%c\r\n", type];
    }
    NSData *sendData = [data dataUsingEncoding:NSASCIIStringEncoding];
    
    
    ASR033WResultCode res = [self sendData:sendData responsePacketType:ASR033WPacketResSetProperty responseSubType:(uint8_t)type];
    if (res != ASR033WResultNoError) {
        return res;
    }
    
    return ASR033WResultNoError;
}

- (ASR033WResultCode)sendCommand:(NSData *)data {
    return [self sendData:data];
}

- (ASR033WResultCode)command:(ASR033WCommandType)command withParameter:(NSString*)param {
    NSString *data = @"";
    if (param != nil && ![param isEqualToString:@""] && param.length > 0) {
        data = [NSString stringWithFormat:@"~a%c%@\r\n", command, param];
    } else {
        data = [NSString stringWithFormat:@"~a%c\r\n", command];
    }
    NSData *sendData = [data dataUsingEncoding:NSASCIIStringEncoding];
    return [self sendData:sendData];
}

- (ASR033WResultCode)command:(ASR033WCommandType)command withParameterData:(NSData *)param{
    NSMutableData *sendData = [[NSMutableData alloc] initWithData:[[NSString stringWithFormat:@"~a%c", command] dataUsingEncoding:NSASCIIStringEncoding]];
    if (param && param.length > 0) {
        [sendData appendData:param];
    }
    NSData *endData = [@"\r\n" dataUsingEncoding:NSASCIIStringEncoding];
    [sendData appendData:endData];
    return [self sendData:sendData];
}

- (ASR033WResultData *)sendDataWithResponse:(NSData *)data responsePacketType:(ASR033WPacketType)resType responseSubType:(uint8_t)subType {
    
    ASR033WPacket *packet = [[ASR033WPacket alloc]initWithPacketType:resType subType:subType];
    
    [mPackets addPacket:packet];
    if (_asr033w.isConnected) {
        [self sendData:data];
        [NSThread detachNewThreadSelector:@selector(changePacket:) toTarget:self withObject:packet];
        while (!packet.data) {
            if (!_asr033w.isConnected){
                break;
            }
            [NSThread sleepForTimeInterval:0.01];
            [[NSRunLoop currentRunLoop] runMode:NSDefaultRunLoopMode beforeDate:[NSDate distantFuture]];
        }
        [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(changePacket:) object:packet];
    }
    
    ASR033WResultCode resCode = (ASR033WResultCode)packet.errorCode;
    NSData *resData = packet.data;
    
    [mPackets removePacket:packet];
    
    if (resCode != ASR033WResultNoError) {
        [NSThread sleepForTimeInterval:0.1];
        if (delegate &&[delegate respondsToSelector:@selector(receivedASR033WDeviceState:)]) {
            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                [self->delegate receivedASR033WDeviceState:resCode];
            }];
        }
    }
    ASR033WResultData *result = [[ASR033WResultData alloc]initWithResultCode:resCode data:resData];
    return result;
}

- (void)changePacket:(ASR033WPacket *)packet{
    [NSThread sleepForTimeInterval:2.0];
    [self performSelectorOnMainThread:@selector(cPacket:) withObject:packet waitUntilDone:NO];
}

- (void)cPacket:(ASR033WPacket *)packet{
    if (!packet.data) {
        packet.data = [[NSData alloc] init];
        packet.errorCode = ASR033WResultTimeout;
    }
}

- (ASR033WResultCode)sendData:(NSData *)data responsePacketType:(ASR033WPacketType)resType responseSubType:(uint8_t)subType {
    
    ASR033WPacket *packet = [[ASR033WPacket alloc]initWithPacketType:resType subType:subType];
    [mPackets addPacket:packet];
    if (_asr033w.isConnected) {
        [self sendData:data];
        [NSThread detachNewThreadSelector:@selector(changePacket:) toTarget:self withObject:packet];
        while (!packet.data) {
            if (!_asr033w.isConnected){
                break;
            }
            [NSThread sleepForTimeInterval:0.01];
            [[NSRunLoop currentRunLoop] runMode:NSDefaultRunLoopMode beforeDate:[NSDate distantFuture]];
        }
        [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(changePacket:) object:packet];
    }
    
    ASR033WResultCode resCode = (ASR033WResultCode)packet.errorCode;
    
    [mPackets removePacket:packet];
    
    if (resCode != ASR033WResultNoError) {
        [NSThread sleepForTimeInterval:0.1];
        if (delegate && [delegate respondsToSelector:@selector(receivedASR033WDeviceState:)]) {
            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                [self->delegate receivedASR033WDeviceState:resCode];
            }];
        }
    }
    return resCode;
}

#pragma mark -  Methods;
- (void)readData:(NSData *)data {
    if (delegate && [delegate respondsToSelector:@selector(receivedASR033WData:)]) {
        [[NSOperationQueue mainQueue] addOperationWithBlock:^{
            [self->delegate receivedASR033WData:data];
        }];
    }
    [mRecvData appendData:data];
    [self readRFProtocol];
}
- (ASR033WResultCode)sendData:(NSData *)data {
//    [[[NSOperationQueue alloc] init] addOperationWithBlock:^{
        if (self->_connectMethod == ASR033WConnectMethod_TCP) {
            if (self->_currentTcp.isConnected) {
                [self->_currentTcp sendData:data];
                return ASR033WResultNoError;
            } else {
                return ASR033WResultNotConnected;
            }
        } else {
            if (self->_currentMQTT.isConnected) {
                [self->_currentMQTT sendHIDORRFIDData:data];
                return ASR033WResultNoError;
            } else {
                return ASR033WResultNotConnected;

            }
        }
//    }];
}
#pragma mark Receive Data Thread Procedure
- (void) readRFProtocol {
    @try {
        NSData* recvData = nil;
        NSRange range;
        while(true)
        {
            range = [ASR033WUtility findDataWidthRange:mRecvData withPrefix:PACKET_PREFIX withSuffix1:PACKET_SUFFIX1 withSuffix2:PACKET_SUFFIX2];
            if (range.location == NSNotFound && range.length == NSNotFound) {
                return;
            }
            
            if (range.location != NSNotFound && range.length != NSNotFound) {
                if (range.length < 6) {
                    [mRecvData replaceBytesInRange:range withBytes:NULL length:0];
                    return;
                } else {
                    recvData = [mRecvData subdataWithRange:NSMakeRange(range.location + 1, range.length - 3)];
                    [mRecvData replaceBytesInRange:range withBytes:NULL length:0];
                }
            } else if (range.location != NSNotFound && range.length == NSNotFound) {
                if (range.location > 0) {
                    range.length = range.location - 1;
                    range.location = 0;
                    [mRecvData replaceBytesInRange:range withBytes:NULL length:0];
                }
                return;
            } else if (range.location == NSNotFound && range.length != NSNotFound) {
                range.location = 0;
                [mRecvData replaceBytesInRange:range withBytes:NULL length:0];
                return;
            } else {
                return;
            }
            
            uint8_t* pack = (uint8_t*)[recvData bytes];
            ASR033WPacketType packType = (ASR033WPacketType)pack[0];
            uint8_t subPack = pack[1];
            NSInteger errorCode = 0;
            int offset = 2;
            int length = (int)[recvData length] - offset;
            
            if (packType != ASR033WPacketEvent) {
                errorCode = [self getErrorCode:recvData findPosition:2];
            }
            NSData *value = nil;
            if (recvData.length > 2) {
                value = [recvData subdataWithRange:NSMakeRange(offset + 4, recvData.length - offset - 4)];
            }
            if (packType == ASR033WPacketEvent) {
                ASR033WEventType eventType = (ASR033WEventType)subPack;
                switch (eventType) {
                    case ASR033WEventTagValue:
                        if (delegate && [delegate respondsToSelector:@selector(receivedASR033WInventoryPC:epc:tid:ant:rssi:phase:frequency:)]) {
                            double rssi = 0;
                            int phase = 0;
                            double frequency = 0;
                            int len = pack[offset] & 0xFF;
                            offset++;
                            int antenna = pack[offset] & 0xFF;
                            switch (antenna) {
                                case 1:
                                    antenna = 1;
                                    break;
                                case 2:
                                    antenna = 2;
                                    break;
                                case 4:
                                    antenna = 3;
                                    break;
                                case 8:
                                    antenna = 4;
                                    break;
                                case 16:
                                    antenna = 5;
                                    break;
                                case 32:
                                    antenna = 6;
                                    break;
                                case 64:
                                    antenna = 7;
                                    break;
                                case 128:
                                    antenna = 8;
                                    break;
                                default:
                                    break;
                            }
                            
                            offset++;
                            int pclength = (pack[offset] &0xF8 )>> 3;
                            
                            NSString * pc = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset, 2)]]];
                            offset = offset + 2;
                            NSLog(@"QQQ offset = %d   pclength = %d   recvData.length = %lu  recvData = %@", offset, pclength * 2, (unsigned long)recvData.length, [ASR033WUtility getHexString:recvData]);
                            NSString  * epc = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset, pclength * 2)]]];
                            offset = offset + pclength * 2;
                            
                            NSData * rssiData = [recvData subdataWithRange:NSMakeRange(offset, 3)];
                            NSString * rssiString = [[NSString alloc] initWithData:rssiData encoding:NSASCIIStringEncoding];
                            rssi = [rssiString intValue];
                            offset = offset + 3;
                            
                            phase = (pack[offset] & 0xFF) / 128.0 * 360.0;
                            offset++;
                            
                            UInt32 tempFrequency = (pack[offset]<<24)+(pack[offset + 1]<<16)+(pack[offset + 2]<<8)+pack[offset + 3];
                            frequency = tempFrequency/1000.0;
                            offset = offset + 4;
                            NSString * tid = @"";
                            if (len > offset + 2) {
                                int tidLength = (pack[offset] & 0xFF << 8) + pack[offset + 1] & 0xFF;
                                offset = offset + 2;
                                tid = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset,tidLength)]]];
                            }
                            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                                [self->delegate receivedASR033WInventoryPC:pc epc:epc tid:tid ant:antenna rssi:rssi phase:phase frequency:frequency];
                            }];
                            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                                [self setLockScreen];
                            }];
                        }
                        break;
                    case ASR033WEventAccessResult: {
                        
                    }
                        break;
                    case ASR033WEventHTTagValue:{
                    }
                        break;
                        
                    default:
                        break;
                }
            } else if (packType == ASR033WPacketCommandAck) {
                mAction = (ASR033WCommandType)subPack;
                if ((mAction == ASR033WCommandStop) || (mAction == ASR033WCommandInventory) || (mAction == ASR033WCommandWriteMemory) || (mAction == ASR033WCommandKill) || (mAction == ASR033WCommandLock)) {
                    ASR033WPacket *packet = [mPackets findPacket:packType subType:subPack];
                    if (packet != nil) {
                        packet.errorCode = errorCode;
                        packet.data = value;
                        [packet signal];
                    }
                    if (delegate && [delegate respondsToSelector:@selector(receivedASR033WActionState:resultCode:)]) {
                        [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                            [self->delegate receivedASR033WActionState:self->mAction resultCode:(ASR033WResultCode)errorCode];
                        }];
                    }
                }
                if (mAction == ASR033WCommandReadMemory) {
                    if (length > RESULT_CODE_LENGTH) {
                        errorCode = [self getErrorCode:recvData findPosition:2];
                    }
                    double rssi = 0;
                    int phase = 0;
                    double frequency = 0;
                    int len = pack[offset] & 0xFF;
                    offset++;
                    int antenna = pack[offset] & 0xFF;
                    switch (antenna) {
                        case 1:
                            antenna = 1;
                            break;
                        case 2:
                            antenna = 2;
                            break;
                        case 4:
                            antenna = 3;
                            break;
                        case 8:
                            antenna = 4;
                            break;
                        case 16:
                            antenna = 5;
                            break;
                        case 32:
                            antenna = 6;
                            break;
                        case 64:
                            antenna = 7;
                            break;
                        case 128:
                            antenna = 8;
                            break;
                        default:
                            break;
                    }
                    
                    offset++;
                    int pclength = (pack[offset] &0xF8 )>> 3;
                    NSString * pc = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset, 2)]]];
                    offset = offset + 2;
                    NSString  * epc = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset, pclength * 2)]]];
                    offset = offset + pclength * 2;
                    NSData * rssiData = [recvData subdataWithRange:NSMakeRange(offset, 3)];
                    NSString * rssiString = [[NSString alloc] initWithData:rssiData encoding:NSASCIIStringEncoding];
                    rssi = [rssiString intValue];
                    offset = offset + 3;
                    phase = (pack[offset] & 0xFF) / 128.0 * 360.0;
                    offset++;
                    UInt32 tempFrequency = (pack[offset]<<24)+(pack[offset + 1]<<16)+(pack[offset + 2]<<8)+pack[offset + 3];
                    frequency = tempFrequency/1000.0;
                    offset = offset + 4;
                    NSData * rssiData1 = [recvData subdataWithRange:NSMakeRange(offset, 4)];
                    NSString *statusCode = [[NSString alloc] initWithData:rssiData1 encoding:NSASCIIStringEncoding];
                    unsigned res = 0;
                    [[NSScanner scannerWithString:statusCode] scanHexInt:&res];
                    offset = offset + 4;
                    NSString * tid = @"";
                    if (len > offset + 2) {
                        int tidLength = (pack[offset] & 0xFF << 8) + pack[offset + 1] & 0xFF;
                        offset = offset + 2;
                        tid = [[NSString alloc] initWithString:[ASR033WUtility getHexString:[recvData subdataWithRange:NSMakeRange(offset,tidLength)]]];
                    }
                    if (delegate && [delegate respondsToSelector:@selector(accessASR033WResult:actionState:epc:data:rssi:phase:frequency:)]) {
                        [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                            [self->delegate accessASR033WResult:(ASR033WResultCode)res actionState:self->mAction epc:epc data:tid rssi:rssi phase:phase frequency:frequency];
                        }];
                    }
                } 
            } else if (packType == ASR033WPacketHID) {
                offset += 1;
                offset += 2;
                NSData * barcodeData = [recvData subdataWithRange:NSMakeRange(3, recvData.length - 3)];
                if (delegate && [delegate respondsToSelector:@selector(receivedASR033WHIDData:)]) {
                    [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                        [self->delegate receivedASR033WHIDData:barcodeData];
                    }];
                }
            }
            else {
                ASR033WPacket *packet = [mPackets findPacket:packType subType:subPack];
                if (packet != nil) {
                    packet.errorCode = errorCode;
                    packet.data = value;
                    [packet signal];
                } else {
                }
            }
        }
    } @catch (NSException *exception) {
        NSLog(@"QQQ exception === %@", exception.reason);
    } @finally {
        
    }
}

#pragma mark - check watcher thread
- (NSInteger)getErrorCode:(NSData *)data findPosition:(int)position {
    int length = 4;
    if ( (int)[data length] - position < 4) {
        length = (int)[data length] - position;
    }
    NSRange range = NSMakeRange(position, length);
    NSData *value = [data subdataWithRange:range];
    NSString *convValue = [[NSString alloc]initWithData:value encoding:NSASCIIStringEncoding];
    unsigned res = 0;
    [[NSScanner scannerWithString:convValue]scanHexInt:&res];
    return res;
} 

- (void)setLockScreen{
    [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(setLockScreenStatus:) object:@(NO)];
    [UIApplication sharedApplication].idleTimerDisabled = YES;
    [self performSelector:@selector(setLockScreenStatus:) withObject:@(NO) afterDelay:0.5];
}

- (void)setLockScreenStatus:(NSNumber *)status{
    [UIApplication sharedApplication].idleTimerDisabled = [status boolValue];
}

- (NSString *)analysisCurrentTemperture:(NSString *)temperture andOtherFields:(NSString *)otherFields {
    if (temperture == nil && otherFields == nil) {
        return nil;
    }
    NSInteger c = [self getDecimalByBinary:[self getBinaryByhex:temperture]];
    NSString *tempertureStr;
    NSArray *bitLabel = [otherFields componentsSeparatedByString:@","];
    if (bitLabel.count != 0) {
        [self stop];
    }
    NSString * nineWord   = [self getBinaryByhex:bitLabel[1]];
    NSString * tenWord    = [self getBinaryByhex:bitLabel[2]];
    NSString * elevenWord = [self getBinaryByhex:bitLabel[3]];
    NSString *wholeStr = [NSString stringWithFormat:@"%@%@%@",nineWord,tenWord,elevenWord];
    NSInteger code1 = [self getDecimalByBinary:[wholeStr substringToIndex:12]];
    NSInteger temp1 = [self getDecimalByBinary:[wholeStr substringWithRange:NSMakeRange(12, 11)]];
    NSInteger code2 = [self getDecimalByBinary:[wholeStr substringWithRange:NSMakeRange(23, 12)]];
    NSInteger temp2 = [self getDecimalByBinary:[wholeStr substringWithRange:NSMakeRange(35, 11)]];
    if (c != 0 && code1 != 0 && temp1 != 0 && code2 != 0 && temp2 != 0) {
        CGFloat tFloat = (CGFloat)(temp2 - temp1)/(code2-code1)*(c-code1);
        tempertureStr = [NSString stringWithFormat:@"%.1f℃",(tFloat+temp1-800)*0.1];
    }
    return tempertureStr;
}
- (NSString *)analysisCurrentdMoisture:(NSString *)moisture {
    NSString *moistureStr;
    if ( [self getDecimalByBinary:[self getBinaryByhex:moisture]] < 10) {
        moistureStr = @"WET";
    } else {
        moistureStr = @"DRY";
    }
    return moistureStr;
}
-(NSString *)getBinaryByhex:(NSString *)hex {
    NSMutableDictionary  *hexDic = [[NSMutableDictionary alloc] init];
    hexDic = [[NSMutableDictionary alloc] initWithCapacity:16];
    [hexDic setObject:@"0000" forKey:@"0"];
    [hexDic setObject:@"0001" forKey:@"1"];
    [hexDic setObject:@"0010" forKey:@"2"];
    [hexDic setObject:@"0011" forKey:@"3"];
    [hexDic setObject:@"0100" forKey:@"4"];
    [hexDic setObject:@"0101" forKey:@"5"];
    [hexDic setObject:@"0110" forKey:@"6"];
    [hexDic setObject:@"0111" forKey:@"7"];
    [hexDic setObject:@"1000" forKey:@"8"];
    [hexDic setObject:@"1001" forKey:@"9"];
    [hexDic setObject:@"1010" forKey:@"A"];
    [hexDic setObject:@"1011" forKey:@"B"];
    [hexDic setObject:@"1100" forKey:@"C"];
    [hexDic setObject:@"1101" forKey:@"D"];
    [hexDic setObject:@"1110" forKey:@"E"];
    [hexDic setObject:@"1111" forKey:@"F"];
    NSString *binaryString=@"";
    for (int i=0; i<[hex length]; i++) {
        NSRange rage;
        rage.length = 1;
        rage.location = i;
        NSString *key = [hex substringWithRange:rage];
        binaryString = [NSString stringWithFormat:@"%@%@",binaryString,[NSString stringWithFormat:@"%@",[hexDic objectForKey:key]]];
    }
    return binaryString;
}
- (NSInteger)getDecimalByBinary:(NSString *)binary {
    NSInteger decimal = 0;
    for (int i=0; i<binary.length; i++) {
        NSString *number = [binary substringWithRange:NSMakeRange(binary.length - i - 1, 1)];
        if ([number isEqualToString:@"1"]) {
            decimal += pow(2, i);
        }
    }
    return decimal;
}
- (ASR033WResultCode)stop {
    return [self command:ASR033WCommandStop withParameter:@""];
}
@end
