#import "ASR033WPacket.h"
#import "ASR033WDefine.h"

#define MAX_TIMEOUT             5

@implementation ASR033WPacket{
    ASR033WPacketType mPacketType;
    uint8_t mSubType;
    NSCondition *mSync;
    int mTimeoutCount;
}

- (id)initWithPacketType:(ASR033WPacketType)type subType:(uint8_t)subType {
    self = [super init];
    if (self) {
        mPacketType = type;
        mSubType = subType;
        self.errorCode = 0;
        self.data = nil;
        mSync = [[NSCondition alloc]init];
        mTimeoutCount = 0;
    }
    return self;
}

- (ASR033WPacketType)type {
    return mPacketType;
}

- (uint8_t)subType {
    return mSubType;
}

- (void)signal {
    [mSync lock];
    [mSync signal];
    [mSync unlock];
}

- (void)wait {
    mTimeoutCount = 0;
    
    [mSync lock];
    [mSync wait];
    [mSync unlock];
}

- (BOOL)equalPacket:(ASR033WPacketType)type subType:(uint8_t)subType {
    if (mPacketType == type && mSubType == subType)
        return YES;
    return NO;
}

- (BOOL)checkTimeoutInterval:(int)interval withTimeout:(int)maxTimeout {
    mTimeoutCount = mTimeoutCount + interval;
    
    
    if (mTimeoutCount >= maxTimeout) {
        return YES;
    }
    return NO;
}

@end


@implementation ASR033WPacketManager{
    NSMutableArray *mPackets;
}

- (id)init {
    self = [super init];
    if (self) {
        mPackets = [[NSMutableArray alloc]init];
    }
    return self;
}

- (void)addPacket:(ASR033WPacket *)packet {
    @synchronized(mPackets) {
        [mPackets addObject:packet];
    }
}

- (void)waitPacket:(ASR033WPacket *)packet {
    if ([mPackets containsObject:packet]) {
        [packet wait];
    } else {
    }
}

- (ASR033WPacket *)waitPacket:(ASR033WPacketType)type subType:(uint8_t)subType {
    ASR033WPacket *packet = [[ASR033WPacket alloc]initWithPacketType:type subType:subType];
    @synchronized(mPackets) {
        [mPackets addObject:packet];
    }
    [packet wait];
    return packet;
}

- (ASR033WPacket *)findPacket:(ASR033WPacketType)type subType:(uint8_t)subType {
    @synchronized(mPackets) {
        for (ASR033WPacket *packet in mPackets) {
            if ([packet equalPacket:type subType:subType] == YES)
                return packet;
        }
    }
    return nil;
}

- (ASR033WPacket *)firstPacket {
    ASR033WPacket * pack = nil;
    
    @synchronized(mPackets) {
        if (mPackets.count <= 0)
            return nil;
        pack = [mPackets firstObject];
    }
    return pack;
}

- (void)removePacket:(ASR033WPacket *)packet {
    @synchronized(mPackets) {
        [mPackets removeObject:packet];
    }
}

- (void)checkTimeoutPacket:(int)interval {
    @synchronized(mPackets) {
        for (ASR033WPacket *packet in mPackets) {
            if ([packet checkTimeoutInterval:interval withTimeout:MAX_TIMEOUT] == YES) {
                packet.errorCode = ASR033WResultTimeout;
                [packet signal];
            }
        }
    }
}

- (BOOL)isWaitPacket {
    if ([mPackets count] > 0) {
        return  YES;
    } else {
        return  NO;
    }
}
@end
