//
//  ASR033WPrinter.m
//  ASR033WSDK
//
//  Created by DS.Zhang on 2019/2/20.
//  Copyright © 2019年 Robin. All rights reserved.
//
#import "ASR033WPrinter.h"
#import "ASR033WNetworkTCP.h"
#import "ASR033WUtility.h"

#define QR_DEFAULT_SIZE   70

typedef NS_ENUM (NSUInteger, CommandType){
    CommandType_NoData = 0,
    CommandType_OpenDrawer,
    CommandType_PrintStatus,
    CommandType_DoPrint,
    CommandType_UserDefine,
};

typedef NS_ENUM (NSUInteger, ASR033WPrinterPIXELS){
    ASR033WPrinterPIXELSALPHA = 0,
    ASR033WPrinterPIXELSBLUE = 1,
    ASR033WPrinterPIXELSGREEN = 2,
    ASR033WPrinterPIXELSRED = 3
};

@interface ASR033WPrinter ()<ASR033WNetworkTCPDelegate>
@property(nonatomic,strong)NSMutableData *commandData;
@end

@implementation ASR033WPrinter{
    ASR033WNetworkTCP *printTcp;
    NSString* host;
    int port;
    CommandType _commandType;
    ASR033WPrinterStatus _printerStatus;
}

#pragma mark - init
-(id)initWithASR033W:(ASR033W *)asr033w {
    self = [super init];
    if (self) {
        host = asr033w.host;
        port = 9100;
        
        _commandData = [[NSMutableData alloc] init];
        
        Byte reset[] = {0x1B,0x40};
        [_commandData appendBytes:reset length:2];
        
        Byte setup[] = {0x1C, 0x43, 0x01};
        [_commandData appendBytes:setup length:3];
        
        _commandType = CommandType_NoData;
        _printerStatus = ASR033WPrinterStatus_NOERROR;
    }
    return self;
}

#pragma mark - method
-(void)getPrinterStatus{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    if (_commandType != CommandType_NoData) {
        return;
    }
    _commandType = CommandType_PrintStatus;
    [self reconnect];
}

-(void)openDrawer{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    if (_commandType != CommandType_NoData) {
        return;
    }
    _commandType = CommandType_OpenDrawer;
    [self reconnect];
}

-(void)doPrint{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    if (_commandType != CommandType_NoData) {
        return;
    }
    _commandType = CommandType_DoPrint;
    [self reconnect];
}
-(void)txRawData:(NSData *)txRawData{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s txRawData = [%@]",__PRETTY_FUNCTION__,[ASR033WUtility dataToString:txRawData]]];
    if (_commandType != CommandType_NoData) {
        return;
    }
    if(printTcp.isConnected){
        [printTcp disconnect];
        [NSThread sleepForTimeInterval:1];
    }
    _commandType = CommandType_UserDefine;
    if(!self.commandData){
        self.commandData = [[NSMutableData alloc] init];
    }
    [self.commandData appendData:txRawData];
    [self reconnect];
}

- (void)reconnect{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    if (!printTcp) {
        if (!host || [@"" isEqualToString:host]) {
            _commandType = CommandType_NoData;
            return;
        }
        printTcp = [[ASR033WNetworkTCP alloc]init];
        [printTcp connect:host port:port];
        [printTcp setDelegate:self];
    }else{
        if(printTcp.isConnected){
            [printTcp disconnect];
            [NSThread sleepForTimeInterval:1];
        }
        [printTcp reconnect];
    }
}
#pragma mark - Instruction

-(void)addCutCommand:(ASR033WPrintCutType)cutType{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s cutType = [%lu]",__PRETTY_FUNCTION__,(unsigned long)cutType]];
    Byte cut[] = {0x1B, cutType};
    [self.commandData appendBytes:cut length:2];
}

-(void)cleanPrintData{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    _commandData = nil;
    _commandData = [[NSMutableData alloc] init];
    
    Byte reset[] = {0x1B,0x40};
    [_commandData appendBytes:reset length:2];
    
    Byte setup[] = {0x1C, 0x43, 0x01};
    [_commandData appendBytes:setup length:3];
}
-(void)addPrintText:(NSString*)text
               font:(ASR033WPrintFont)font
   printerAlignment:(ASR033WPrintAlignment)printerAlignment{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s text = [%@] font = [%lu] printerAlignment = [%lu]",__PRETTY_FUNCTION__,text,(unsigned long)font,(unsigned long)printerAlignment]];
    Byte alig[] = {0x1B,0x61,printerAlignment};
    [self.commandData appendBytes:alig length:3];
    
    Byte textFont[] = {0x1D,0x21,font};
    [self.commandData appendBytes:textFont length:3];
    
    NSStringEncoding enc = CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingDOSJapanese);
    NSData *data = [text dataUsingEncoding:enc];
    [self.commandData appendData:data];
    
    Byte reset[] = {0x0A};
    [self.commandData appendBytes:reset length:1];
}
-(void)addPrintQRCode:(NSString*)qrCode
                 size:(ASR033WPrintQrDotSize)size
     printerAlignment:(ASR033WPrintAlignment)printerAlignment{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s qrCode = [%@] size = [%lu] printerAlignment = [%lu]",__PRETTY_FUNCTION__,qrCode,(unsigned long)size,(unsigned long)printerAlignment]];

    UIImage *dataImage = [[UIImage alloc] init];
    float qrSize = size * QR_DEFAULT_SIZE;
    if (qrCode && ![qrCode isEqualToString:@""]) {
        dataImage = [self createNonInterpolatedUIImageFormCIImage:[self createQRForString:qrCode] withSize:qrSize];
        if (dataImage) {
            [self addPrintImage:dataImage
                          width:qrSize
                         height:qrSize
               printerAlignment:printerAlignment];
        }
    }
}
-(void)addPrintImage:(UIImage *)image
               width:(float)width
              height:(float)height
    printerAlignment:(ASR033WPrintAlignment)printerAlignment{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    Byte alig[] = {0x1B,0x61,printerAlignment};
    [self.commandData appendBytes:alig length:3];
    
    UIImage *printImage = [self scaleImageWithImage:image width:width height:height];
    NSData *data = [self imageToThermalData:printImage];
    [self.commandData appendData:data];
    
    Byte line[] = {0x1B, 0x4A, 50};
    [self.commandData appendBytes:line length:3];
}

//调整分辨率
- (UIImage *)scaleImageWithImage:(UIImage *)image width:(NSInteger)width height:(NSInteger)height{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    CGSize size;
    size.width = width;
    size.height = height;
    UIGraphicsBeginImageContext(size);
    [image drawInRect:CGRectMake(0, 0, width, height)];
    UIImage *scaledImage = UIGraphicsGetImageFromCurrentImageContext();
    UIGraphicsEndImageContext();
    return scaledImage;
}

//位图模式指令
//根据像素信息将图片进行黑白化处理，并逐行拼接打印信息
-(NSData *) imageToThermalData:(UIImage*)image {
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    CGImageRef imageRef = image.CGImage;
    // Create a bitmap context to draw the uiimage into
    CGContextRef context = [self CreateARGBBitmapContextWithCGImageRef:imageRef];
    if(!context) {
        return NULL;
    }
    
    size_t width = CGImageGetWidth(imageRef);
    size_t height = CGImageGetHeight(imageRef);
    
    CGRect rect = CGRectMake(0, 0, width, height);
    
    // Draw image into the context to get the raw image data
    CGContextDrawImage(context, rect, imageRef);
    
    // Get a pointer to the data
    uint32_t *bitmapData = (uint32_t *)CGBitmapContextGetData(context);
    
    if(bitmapData) {
        
        uint8_t *m_imageData = (uint8_t *) malloc(width * height/8 + 8*height/8);
        memset(m_imageData, 0, width * height/8 + 8*height/8);
        int result_index = 0;
        
        for(int y = 0; (y + 24) < height;) {
            m_imageData[result_index++] = 27;
            m_imageData[result_index++] = 51;
            m_imageData[result_index++] = 0;
            
            m_imageData[result_index++] = 27;
            m_imageData[result_index++] = 42;
            m_imageData[result_index++] = 33;
            
            m_imageData[result_index++] = width%256;
            m_imageData[result_index++] = width/256;
            for(int x = 0; x < width; x++) {
                int value = 0;
                for (int temp_y = 0 ; temp_y < 8; ++temp_y)
                {
                    uint8_t *rgbaPixel = (uint8_t *) &bitmapData[(y+temp_y) * width + x];
                    
                    uint32_t gray = 0.3 * rgbaPixel[ASR033WPrinterPIXELSRED] + 0.59 * rgbaPixel[ASR033WPrinterPIXELSGREEN] + 0.11 * rgbaPixel[ASR033WPrinterPIXELSBLUE];
                    if (gray < 127)
                    {
                        value += 1<<(7-temp_y)&255;
                    }
                }
                m_imageData[result_index++] = value;
                
                value = 0;
                for (int temp_y = 8 ; temp_y < 16; ++temp_y)
                {
                    uint8_t *rgbaPixel = (uint8_t *) &bitmapData[(y+temp_y) * width + x];
                    uint32_t gray = 0.3 * rgbaPixel[ASR033WPrinterPIXELSRED] + 0.59 * rgbaPixel[ASR033WPrinterPIXELSGREEN] + 0.11 * rgbaPixel[ASR033WPrinterPIXELSBLUE];
                    
                    if (gray < 127)
                    {
                        value += 1<<(7-temp_y%8)&255;
                    }
                    
                }
                m_imageData[result_index++] = value;
                
                value = 0;
                for (int temp_y = 16 ; temp_y < 24; ++temp_y)
                {
                    uint8_t *rgbaPixel = (uint8_t *) &bitmapData[(y+temp_y) * width + x];
                    uint32_t gray = 0.3 * rgbaPixel[ASR033WPrinterPIXELSRED] + 0.59 * rgbaPixel[ASR033WPrinterPIXELSGREEN] + 0.11 * rgbaPixel[ASR033WPrinterPIXELSBLUE];
                    
                    if (gray < 127)
                    {
                        value += 1<<(7-temp_y%8)&255;
                    }
                    
                }
                m_imageData[result_index++] = value;
            }
            m_imageData[result_index++] = 13;
            m_imageData[result_index++] = 10;
            y += 24;
        }
        NSMutableData *data = [[NSMutableData alloc] initWithCapacity:0];
        [data appendBytes:m_imageData length:result_index];
        free(bitmapData);
        return data;
        
    } else {
        NSLog(@"Error getting bitmap pixel data\n");
    }
    
    CGContextRelease(context);
    
    return nil ;
}

//获取像素数据
- (CGContextRef)CreateARGBBitmapContextWithCGImageRef:(CGImageRef)inImage{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    CGContextRef    context = NULL;
    CGColorSpaceRef colorSpace;
    void *          bitmapData;
    int             bitmapByteCount;
    int             bitmapBytesPerRow;
    
    // Get image width, height. We'll use the entire image.
    size_t pixelsWide = CGImageGetWidth(inImage);
    size_t pixelsHigh = CGImageGetHeight(inImage);
    
    // Declare the number of bytes per row. Each pixel in the bitmap in this
    // example is represented by 4 bytes; 8 bits each of red, green, blue, and
    // alpha.
    bitmapBytesPerRow   = (int)(pixelsWide * 4);
    bitmapByteCount     = (int)(bitmapBytesPerRow * pixelsHigh);
    
    // Use the generic RGB color space.
    colorSpace =CGColorSpaceCreateDeviceRGB();
    if (colorSpace == NULL)
    {
        return NULL;
    }
    
    // Allocate memory for image data. This is the destination in memory
    // where any drawing to the bitmap context will be rendered.
    bitmapData = malloc( bitmapByteCount );
    if (bitmapData == NULL)
    {
        CGColorSpaceRelease( colorSpace );
        return NULL;
    }
    
    // Create the bitmap context. We want pre-multiplied ARGB, 8-bits
    // per component. Regardless of what the source image format is
    // (CMYK, Grayscale, and so on) it will be converted over to the format
    // specified here by CGBitmapContextCreate.
    context = CGBitmapContextCreate (bitmapData,
                                     pixelsWide,
                                     pixelsHigh,
                                     8,      // bits per component
                                     bitmapBytesPerRow,
                                     colorSpace,
                                     kCGImageAlphaPremultipliedFirst);
    if (context == NULL)
    {
        free (bitmapData);
    }
    
    // Make sure and release colorspace before returning
    CGColorSpaceRelease( colorSpace );
    
    return context;
}

#pragma mark - ASR033WNetworkTCPDelegate
- (void)onTCPReceive:(NSData *)data{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s data = [%@]",__PRETTY_FUNCTION__,[ASR033WUtility dataToString:data]]];

    dispatch_queue_t queue = dispatch_queue_create("gcddemo", DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{
        NSString *str = [self convertDataToHexStr:data];
        NSString *string = [self getBinaryByHex:str];
        if(self->_commandType == CommandType_UserDefine){
            self->_commandType = CommandType_NoData;
            [self->printTcp disconnect];
            if (self.printDelegate && [self.printDelegate respondsToSelector:@selector(rxRawData:)]) {
                [self.printDelegate rxRawData:data];
            }
        }else if(self->_commandType == CommandType_PrintStatus){
            if (string && ![string isEqualToString:@""] && string.length == 8) {
                if (self->_printerStatus == ASR033WPrinterStatus_CoverOpen) {
                    NSString *coverOpenStr = [string substringWithRange:NSMakeRange(5, 1)];
                    if (coverOpenStr && [coverOpenStr isEqualToString:@"1"]) {
                        self->_commandType = CommandType_NoData;
                        [self->printTcp disconnect];
                        if (self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterStatus:)]) {
                            [self.printDelegate ASR033WPrinterStatus:ASR033WPrinterStatus_CoverOpen];
                        }
                    } else {
                        self->_printerStatus = ASR033WPrinterStatus_NOPAPER;
                        NSMutableData *data = [[NSMutableData alloc] init];
                        Byte reset[] = {0x1B, 0x76, 0x02};
                        [data appendBytes:reset length:3];
                        [self->printTcp sendData:data];
                    }
                } else if (self->_printerStatus == ASR033WPrinterStatus_NOPAPER) {
                    self->_printerStatus = ASR033WPrinterStatus_NOERROR;
                    self->_commandType = CommandType_NoData;
                    [self->printTcp disconnect];
                    NSString *noPaperStr = [string substringWithRange:NSMakeRange(5, 1)];
                    NSString *institutionalFailureStr = [string substringWithRange:NSMakeRange(4, 1)];
                    if (noPaperStr && [noPaperStr isEqualToString:@"1"]) {
                        if (self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterStatus:)]) {
                            [self.printDelegate ASR033WPrinterStatus:ASR033WPrinterStatus_NOPAPER];
                        }
                    } else if (institutionalFailureStr && [institutionalFailureStr isEqualToString:@"1"]) {
                        if (self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterStatus:)]) {
                            [self.printDelegate ASR033WPrinterStatus:ASR033WPrinterStatus_InstitutionalFailure];
                        }
                    } else {
                        if (self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterStatus:)]) {
                            [self.printDelegate ASR033WPrinterStatus:ASR033WPrinterStatus_NOERROR];
                        }
                    }
                }
            }
        }else{
            self->_commandType = CommandType_NoData;
            [self->printTcp disconnect];
        }
    });
}

- (void)onTCPDisconnect{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    self->_isConnected = NO;
    self->_commandType = CommandType_NoData;
    if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterConnected:)]){
        [self.printDelegate ASR033WPrinterConnected:self->_isConnected];
    }
}

- (void)onTCPConnected{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    dispatch_queue_t queue = dispatch_queue_create("gcddemo", DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{
        self->_isConnected = YES;
        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterConnected:)]){
            [self.printDelegate ASR033WPrinterConnected:self->_isConnected];
        }
        switch (self->_commandType) {
            case CommandType_DoPrint:{
                if ([self commandData] && [self commandData].length > 0) {
                    if (self->_isConnected && self->printTcp.isConnected) {
                        [self->printTcp sendData:self.commandData];
                        [self cleanPrintData];
                        self->_commandType = CommandType_NoData;
                        [self->printTcp disconnect];
                        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(whenPrintSuccessed)]){
                            [self.printDelegate whenPrintSuccessed];
                        }
                    }else{
                        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(retryPrintOnError)]){
                            if ([self.printDelegate retryPrintOnError] ) {
                                self->_commandType = CommandType_DoPrint;
                                [self->printTcp reconnect];
                            }
                        }else{
                            [self cleanPrintData];
                            self->_commandType = CommandType_NoData;
                        }
                    }
                }
            }
                break;
            case CommandType_PrintStatus:{
                self->_printerStatus = ASR033WPrinterStatus_CoverOpen;
                NSMutableData *coverData = [[NSMutableData alloc] init];
                Byte coverReset[] = {0x10, 0x04, 0x02};
                [coverData appendBytes:coverReset length:3];
                [self->printTcp sendData:coverData];
            }
                break;
            case CommandType_OpenDrawer:{
                NSMutableData *data = [[NSMutableData alloc] init];
                Byte open[] = {0x1B, 0x70, 0x00, 0x80, 0xFF};
                [data appendBytes:open length:5];
                [self->printTcp sendData:data];
                self->_commandType = CommandType_NoData;
                [self->printTcp disconnect];
            }
                break;
            case CommandType_NoData:{
                self->_commandType = CommandType_NoData;
            }
                break;
            case CommandType_UserDefine:{
                if ([self commandData] && [self commandData].length > 0) {
                    if (self->_isConnected && self->printTcp.isConnected) {
                        [self->printTcp sendData:self.commandData];
                        [self cleanPrintData];
                        self->_commandType = CommandType_NoData;
                        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(whenPrintSuccessed)]){
                            [self.printDelegate whenPrintSuccessed];
                        }
                    }else{
                        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(retryPrintOnError)]){
                            if ([self.printDelegate retryPrintOnError] ) {
                                self->_commandType = CommandType_UserDefine;
                                [self->printTcp reconnect];
                            }
                        }else{
                            [self cleanPrintData];
                            self->_commandType = CommandType_NoData;
                        }
                    }
                }
            }
                break;
            default:
                self->_commandType = CommandType_NoData;
                break;
        }
    });
}

- (void)onTCPErrorReceive:(NSError *)err{
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s error = [%@]",__PRETTY_FUNCTION__,err]];
    dispatch_queue_t queue = dispatch_queue_create("gcddemo", DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{
        self->_commandType = CommandType_NoData;
        NSString *errReason = [NSString stringWithFormat:@"ErrCode %d : %@",(int)[err code],[err localizedDescription]];
        self->_isConnected = NO;
        if(self.printDelegate && [self.printDelegate respondsToSelector:@selector(ASR033WPrinterConnected:)]){
            [self.printDelegate ASR033WPrinterConnected:self->_isConnected];
        }
    });
}

//NSData转16
- (NSString *)convertDataToHexStr:(NSData *)data {
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    if (!data || [data length] == 0) {
        return @"";
    }
    NSMutableString *string = [[NSMutableString alloc] initWithCapacity:[data length]];
    
    [data enumerateByteRangesUsingBlock:^(const void *bytes, NSRange byteRange, BOOL *stop) {
        unsigned char *dataBytes = (unsigned char*)bytes;
        for (NSInteger i = 0; i < byteRange.length; i++) {
            NSString *hexStr = [NSString stringWithFormat:@"%x", (dataBytes[i]) & 0xff];
            if ([hexStr length] == 2) {
                [string appendString:hexStr];
            } else {
                [string appendFormat:@"0%@", hexStr];
            }
        }
    }];
    return string;
}

//16转2
- (NSString *)getBinaryByHex:(NSString *)hex {
    [ASR033WUtility writeLog:[NSString stringWithFormat:@"%s",__PRETTY_FUNCTION__]];
    NSMutableDictionary *hexDic = [[NSMutableDictionary alloc] initWithCapacity:16];
    [hexDic setObject:@"0000" forKey:@"0"];
    [hexDic setObject:@"0001" forKey:@"1"];
    [hexDic setObject:@"0010" forKey:@"2"];
    [hexDic setObject:@"0011" forKey:@"3"];
    [hexDic setObject:@"0100" forKey:@"4"];
    [hexDic setObject:@"0101" forKey:@"5"];
    [hexDic setObject:@"0110" forKey:@"6"];
    [hexDic setObject:@"0111" forKey:@"7"];
    [hexDic setObject:@"1000" forKey:@"8"];
    [hexDic setObject:@"1001" forKey:@"9"];
    [hexDic setObject:@"1010" forKey:@"A"];
    [hexDic setObject:@"1011" forKey:@"B"];
    [hexDic setObject:@"1100" forKey:@"C"];
    [hexDic setObject:@"1101" forKey:@"D"];
    [hexDic setObject:@"1110" forKey:@"E"];
    [hexDic setObject:@"1111" forKey:@"F"];
    
    NSString *binary = @"";
    for (int i=0; i<[hex length]; i++) {
        
        NSString *key = [hex substringWithRange:NSMakeRange(i, 1)];
        NSString *value = [hexDic objectForKey:key.uppercaseString];
        if (value) {
            
            binary = [binary stringByAppendingString:value];
        }
    }
    return binary;
}

#pragma mark - InterpolatedUIImage
- (UIImage *)createNonInterpolatedUIImageFormCIImage:(CIImage *)image withSize:(CGFloat) size {
    CGRect extent = CGRectIntegral(image.extent);
    CGFloat scale = MIN(size/CGRectGetWidth(extent), size/CGRectGetHeight(extent));
    size_t width = CGRectGetWidth(extent) * scale;
    size_t height = CGRectGetHeight(extent) * scale;
    CGColorSpaceRef cs = CGColorSpaceCreateDeviceGray();
    CGContextRef bitmapRef = CGBitmapContextCreate(nil, width, height, 8, 0, cs, (CGBitmapInfo)kCGImageAlphaNone);
    CIContext *context = [CIContext contextWithOptions:nil];
    CGImageRef bitmapImage = [context createCGImage:image fromRect:extent];
    CGContextSetInterpolationQuality(bitmapRef, kCGInterpolationNone);
    CGContextScaleCTM(bitmapRef, scale, scale);
    CGContextDrawImage(bitmapRef, extent, bitmapImage);
    
    CGImageRef scaledImage = CGBitmapContextCreateImage(bitmapRef);
    // Cleanup
    CGContextRelease(bitmapRef);
    CGImageRelease(bitmapImage);
    return [UIImage imageWithCGImage:scaledImage];
}

#pragma mark - QRCodeGenerator
- (CIImage *)createQRForString:(NSString *)qrString {
    
    NSData *stringData = [qrString dataUsingEncoding:NSUTF8StringEncoding];
    
    CIFilter *qrFilter = [CIFilter filterWithName:@"CIQRCodeGenerator"];
    
    [qrFilter setValue:stringData forKey:@"inputMessage"];
    [qrFilter setValue:@"M" forKey:@"inputCorrectionLevel"];

    return qrFilter.outputImage;
}

#pragma mark - imageToTransparent
void ProviderReleaseData (void *info, const void *data, size_t size){
    free((void*)data);
}
- (UIImage*)imageBlackToTransparent:(UIImage*)image withRed:(CGFloat)red andGreen:(CGFloat)green andBlue:(CGFloat)blue{
    const int imageWidth = image.size.width;
    const int imageHeight = image.size.height;
    size_t      bytesPerRow = imageWidth * 4;
    uint32_t* rgbImageBuf = (uint32_t*)malloc(bytesPerRow * imageHeight);
    
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    CGContextRef context = CGBitmapContextCreate(rgbImageBuf, imageWidth, imageHeight, 8, bytesPerRow, colorSpace, kCGBitmapByteOrder32Little | kCGImageAlphaNoneSkipLast);
    CGContextDrawImage(context, CGRectMake(0, 0, imageWidth, imageHeight), image.CGImage);
    
    int pixelNum = imageWidth * imageHeight;
    uint32_t* pCurPtr = rgbImageBuf;
    for (int i = 0; i < pixelNum; i++, pCurPtr++){
        if ((*pCurPtr & 0xFFFFFF00) < 0x99999900){
            
            uint8_t* ptr = (uint8_t*)pCurPtr;
            ptr[3] = red; //0~255
            ptr[2] = green;
            ptr[1] = blue;
        }else{
            uint8_t* ptr = (uint8_t*)pCurPtr;
            ptr[0] = 0;
        }
    }
    
    CGDataProviderRef dataProvider = CGDataProviderCreateWithData(NULL, rgbImageBuf, bytesPerRow * imageHeight, ProviderReleaseData);
    CGImageRef imageRef = CGImageCreate(imageWidth, imageHeight, 8, 32, bytesPerRow, colorSpace, kCGImageAlphaLast | kCGBitmapByteOrder32Little, dataProvider, NULL, true, kCGRenderingIntentDefault);
    CGDataProviderRelease(dataProvider);
    UIImage* resultUIImage = [UIImage imageWithCGImage:imageRef];
    
    CGImageRelease(imageRef);
    CGContextRelease(context);
    CGColorSpaceRelease(colorSpace);
    return resultUIImage;
}
@end
