#include <iostream>  
#include <thread>  
#include <mutex>  
#include <functional>  
#include <cstring>  
#include <arpa/inet.h>  
#include <unistd.h>  
  
class Client {  
public:  



    using Callback = std::function<void(const std::string&)>;  
  
    Client(const std::string& server_ip, int server_port)  
        : server_ip_(server_ip), server_port_(server_port), running_(false) {}  
  
    ~Client() {  
        stop();  
    }  
  
    bool start(Callback callback) {  
        if (running_) {  
            std::cerr << "Client is already running!" << std::endl;  
            return false;  
        }  
  
        callback_ = callback;  
        sockfd_ = socket(AF_INET, SOCK_STREAM, 0);  
        if (sockfd_ < 0) {  
            std::cerr << "Failed to create socket!" << std::endl;  
            return false;  
        }  
  
        sockaddr_in server_addr;  
        std::memset(&server_addr, 0, sizeof(server_addr));  
        server_addr.sin_family = AF_INET;  
        server_addr.sin_port = htons(server_port_);  
        if (inet_pton(AF_INET, server_ip_.c_str(), &server_addr.sin_addr) <= 0) {  
            std::cerr << "Invalid server IP address!" << std::endl;  
            close(sockfd_);  
            return false;  
        }  
  
        if (connect(sockfd_, (sockaddr*)&server_addr, sizeof(server_addr)) < 0) {  
            std::cerr << "Failed to connect to server!" << std::endl;  
            close(sockfd_);  
            return false;  
        }  
  
        running_ = true;  
        recv_thread_ = std::thread(&Client::recvLoop, this);  
        return true;  
    }  
  
    void stop() {  
        if (running_) {  
            running_ = false;  
            if (recv_thread_.joinable()) {  
                recv_thread_.join();  
            }  
            close(sockfd_);  
        }  
    }  
  
    bool sendMessage(const std::string& message) {  
        if (!running_) {  
            std::cerr << "Client is not running!" << std::endl;  
            return false;  
        }  
        std::cout<<message<<std::endl;
      //   ssize_t bytes_sent = send(sockfd_, message.c_str(), message.size(), 0);  
       auto msg = HexstringToByteGroup(message);
        ssize_t bytes_sent = send(sockfd_, msg.data(),msg.size(), 0);  
        if (bytes_sent < 0) {  
            std::cerr << "Failed to send message!" << std::endl;  
            return false;  
        }  
  
        return true;  
    }  
  
private: 
    std::string server_ip_;  
    int server_port_;  
    int sockfd_ = -1;  
    std::thread recv_thread_;  
    Callback callback_;  
    std::mutex mutex_;  
    bool running_;  

    void recvLoop() 
    {  
        char buffer[1024];  

        std::vector<char> receive_buffer;
        receive_buffer.reserve(99); 

        while (running_) 
        {  
            ssize_t bytes_recv = recv(sockfd_, buffer, sizeof(buffer) - 1, 0);  
            if (bytes_recv > 0) 
            {
                  receive_buffer.insert(receive_buffer.end(), buffer, buffer + bytes_recv);  
                  
                  char second_last_char = receive_buffer[receive_buffer.size() - 2];  
                  char last_char = receive_buffer.back();

                  // if (receive_buffer.size() >= 26) 
                  if (second_last_char == 0x0d && last_char == 0x0a) 
                  {  
                        std::lock_guard<std::mutex> lock(mutex_);  
                        if (callback_) 
                        {
                              callback_(std::string(receive_buffer.begin(), receive_buffer.end()));  
                        }  
                        receive_buffer.clear();  
                  } 
                  else 
                  {  
                        for(char c:receive_buffer)
                        {
                              std::printf("%02X ", c); 
                        }
                        std::cerr << "Received incomplete or corrupted message." << std::endl;  
                        receive_buffer.clear();  
                  }  
            } 
            else if (bytes_recv == 0) 
            {  
                  std::cerr << "Server disconnected!" << std::endl;  
                  running_ = false;  
                  break;  
            } 
            else 
            {  
                std::cerr << "Failed to receive message!" << std::endl;  
                running_ = false;  
                break;  
            }  
        }  
    }  
};  